;-----------------------------------------------------------------------------
;* scatter32.asm                                letzte Aenderung: 9.12.2010
;* wie scatter328.asm aber fuer ATmega32 und mit Grafik-LCD
;* Version: 1.08
;* Autor: Rolf Pfister
;* Copyright: Freeware
;* Prozessor: ATmega32A oder ATmega32-16PU, 8MHz
;* Schaltschema: scatter32a.png oder scatter32b.png
;*	 PortB: B0, B1, B2 mit LEDs verbunden, B4 Basis auf NPN-Transistor
;*	        B3 ueber 470 Ohm mit Taster verbunden, wenn gedrueckt == 0
;*	        B6, B7 weitere Taster (ueber 470 Ohm verbunden)
;*	 PortD:	D2..D7 fuer LCD-Anzeige (sowohl bei Text als auch bei Grafik)
;*	 PortC:	C0..C7 fuer LCD-Anzeige (fuer Grafik-LCD)
;*   Variante 1: ADC0 (Pin 40), Messspannung nach Verstaerkung mit OP
;*
;* History:
;* 20.8.2010	Erstellung aus scatter328.asm, angepasst an ATmega32
;* 14.9.10 1.05	Scheint soweit zu funktionieren
;* 17.9.10 1.06	Alte Grafikpunkte jeweils loeschen. Nur 1 Zeile fuer Text,
;*		somit 56 Punkte Platz fuer Grafik. Mit Skalierfaktor, der
;*		sich waehrend der Messung anpasst.
;* 9.11.10 1.07	Marker fuer aktuelle Grafikposition
;* 10.11.	Marker anders gemacht. Schuettler bei 1 Sek Messperiode jetzt
;*		jede Minute (bisher ohne Schuetteln bei nur 1 Sek)
;* 23.11.  1.08	Erweiterung zum Auslesen def Flashs ueber UART
;* 26.11.	Neu auch noch UV-LED schalten ueber PB5 (scatter32b.png)
;*  8.12.	Noch 2 Tastschalter PB6 PB7 zum UV-Schaltzeiten einstellen
;*
;-----------------------------------------------------------------------------

.include "m32def.inc"		; Definitionsdatei fuer den Prozessortyp
.define ATMEGA32		; ATMEGA8, ATMEGA328P oder ATMEGA32

;; zum Austesten mit ATmega8:
;.include "m8def.inc"		; Definitionsdatei fuer den Prozessortyp
;.define ATMEGA8		; ATMEGA8, ATMEGA328P oder ATMEGA32
;.equ SRAM_START = 0x60
;.equ EEPROMEND  = 0x01ff
;; Ende Austesten mit ATmega8

.equ F_CPU=8000000		; Quarzfrequenz in Herz
.equ	ANZAHLSENSOREN=1	;(vom thermologger uebernommen)
.equ	SENSORTYP=83		;Phototransistor BPX38
.equ	ANSCHLUSSVARIANTE=1	;Verstaerkt mit BC550 oder OP, 10kOhm am Emitt.
;.equ	ANSCHLUSSVARIANTE2=3	;Anschluss des 2.Sensors
.equ	SENSORSPEICHERN=1	;1=1.Sensor, 2=Wert vom 2.Sensor speichern
;.equ	DATENFORMAT=0		;0=ungepackt = 2 Byte pro Messpunkt
.equ	DATENFORMAT=1	;1 oder 2=Einfaches packen = 0.5 bis 2 Byte pro Mess.
;.define EEPROMVERSION	;nur beim ATmega8 getestet
.define FLASHVERSION

.define LCDANZEIGE	;Anzeige des Messwertes auf einem LCD-Modul
;.define TEXT_LCD	;entweder auf einem Text-LCD-Modul
.define GRAFIK_LCD	;oder dann auf einem Grafik-LCD-Modul
.define STACKERROR	;Pruefen auf Stack-Fehler
;.define MAGNETRUEHRER  ;falls Magnetruehrer ein/aus-schalten gebraucht wird
.define SCHUETTLER	;wenn Schuettelmaschine gebraucht wird
;.define MULDIV         ;falls Multiplikation und Division gebraucht wird
.define MARKER		;markieren von momentaner Position im Grafik-LCD
.define SEKSCHUETTLER	;bei 1 Sek Messperiode jede Minute schuetteln
.define TASTMARKER	;Auf Tastendruck Setzen eines Markers in gespeicherten Daten
.define UARTREADOUT	;Auslesen des Flashs ueber UART
.define LEDSCHALTUHR	;Schalten der UV-LED  (statt MAGNETRUEHRER der eh nicht benutz wurde)

;; zur Fehlersuche:
;.define DEBUGGING
;.define TESTLOOP
;.define GLCDTEST
;.define GRAFIKTEST
.define VERSIONSANZEIGE

;; Aendern falls Hardware aendert:
.equ TAST_PIN=PINB		; EingabePort an dem Taster angeschlossen ist
.equ TAST_PORT=PORTB		; Taster-Port fuer Pullup-Widerstand
.equ TASTER=3			; Bit an dem Taster angeschlossen ist
.ifdef ATMEGA32
  .equ TAST_PIN2=PINB		; EingabePort fuer 2. und 3. Taster
  .equ TAST_PORT2=PORTB		; fuer Pullup-Widerstaende
  .equ TASTER2=6		; zweiter Taster
  .equ TASTER3=7		; dritter Taster
.else 
  .equ TAST_PIN2=PINC		; EingabePort fuer 2. und 3. Taster
  .equ TAST_PORT2=PORTC		; fuer Pullup-Widerstaende
  .equ TASTER2=4		; zweiter Taster
  .equ TASTER3=5		; dritter Taster
.endif

;-----------------------------------------------------------------------------
;Datenformat:
; 1 Byte DATENFORMAT
; 1 Byte SENSORTYP
; 1 Byte ANSCHLUSSVARIANTE
; 1 Byte Messperiode = Anzahl Sekunden zwischen 2 Messungen
; 2 Byte N = Anzahl Messpunkte
;Wenn DATENFORMAT==0: N*2 Byte ungepackte Daten
;sonst je nach Packvariante weniger als N*2 Bytes
;
;DATENFORMAT==1:
; Wenn erstes Bit gesetzt, dann sind die naechsten 3 Bits
; die Differenz zum vorherigen Wert (Bereich -4 bis +3).
; Wenn das erste Bit nicht, aber das 2.Bit gesetzt ist,
; dann sind die naechsten 6 Bits +-4 die Differenz zum vorherigen Wert.
; (+-4 heisst: wenn Diff. positiv noch 4 addieren, wenn negativ 4 subtrahieren)
; Somit Wertebereich 4 bis 35 und -5 bis -36.
; Wenn weder das erste noch das 2. Bit gesetzt sind (c & 0xC0 == 0), dann
; sind die naechsten 10 Bits ein absoluter Wert.
; Also kurz zusammengefasst:
;       1X -> 4Bit, davon 3Bit Differenzwert (-4 ... +3)
;       01 -> 8Bit, davon 6Bit Differenzwert (-36 ... -5 und +4 ... +35)
;       00 -> 12Bit, davon 10Bit Absolutwert vorzeichenlos (0 ... 1023)
; Die Werte sind Messwerte von AD-Wandlung.
;
;DATENFORMAT==2:
; Wie 1 aber mit 14Bit-Absolutwerten:
;       00 -> 16Bit, davon 14Bit Absolutwert vorzeichenlos (0 ... 16383)
; Die Werte sind GradCelsius*10 (z.B. 357 bedeutet 35.7 Grad)
;
;-----------------------------------------------------------------------------
;Speicherbelegung:
;Global in Interrupts benutzt:
; r11:r10 Millisekunden, r12 Sekunden, r13 Minuten, r14 Stunden, r15 Tage
; r4=Anzahl Aufsummierungen in AD-Wandler
;Global benutzt:
; r5: Vorheriges Nibble beim gepackten Speichern (Gueltig=r5&0x10)
; r7:r6 letzter Messwert
; r8 = Zwischenspeicher fuer imflashspeichern
; r9 = Flags:
;   r9 Bit0 = Flag fuer erste Page fuer imflashspeichern
;   r9 Bit1 = Nicht-Speichern-Flag (in aktueller Version nicht benutzt)
; r23:r22 Restlicher Platz in Byte
; r25:r24 Anzahl bisher gespeicherte Messpunkte
; r27:r26 == XH:XL = Zeiger auf zu speichernde Daten
; r29:r28 == YH:YL = Zeiger auf am Schluss zu speichernde Anzahl Messpunkte
; 
;im Hauptprogramm:
; r17:r16 fuer kurze Wartezeit (immer innerhalb 1...255)
; r16 auch als temporaere Variable
; r1=Messperiode Anzeige mit LEDs, 1=rot 2=gruen 4=gelb
; r2=Flag fuer Mittelung, 0=ohne, 1=mit
; r18=Sensor der gespeichert und auf LCD angezeigt werden soll
; r19=Messperiode in Sekunden
; r3=1
; r20,r21 Sollzeit
; r31:r30 == ZH:ZL = Zeiger zum umkopieren (in page1zurueckkopieren)
; 
;-----------------------------------------------------------------------------
.equ SRAMSTART=SRAM_START
.equ PAGESIZEB = PAGESIZE*2	; PAGESIZEB is page size in BYTES, not words
; Variablen im SRAM:
		.DSEG
erstepage:	.BYTE PAGESIZEB
lcdpos:		.BYTE 1		;LCD-Position
;magnetschaltzeit: .BYTE 1	;wurde doch nicht benoetigt
;sollwertH:	.BYTE 1		;Sollwert fuer Thermostat
;istwertL:	.BYTE 1	
;istwertH:	.BYTE 1		;Istwert fuer Thermostat
ledschaltzeit:	.BYTE 1		;zum Schalten der UV-LED
ledeinzeit:	.BYTE 1		;Anzahl Minuten wo LED eingeschaltet sein soll
ledauszeit:	.BYTE 1		;Anzahl Minuten wo LED ausgeschaltet sein soll
reserve1:	.BYTE 1
widerstandL:	.BYTE 1
widerstandH:	.BYTE 1
sensor2gradL:	.BYTE 1
sensor2gradH:	.BYTE 1		;Wert von Sensor2 in GradCelsius*10
lcdn0:		.BYTE 1
lcdn1:		.BYTE 1
lcdn2:		.BYTE 1
lcdn3:		.BYTE 1		;Zahlen fuers LCD
sensor1adwertL:	.BYTE 1
sensor1adwertH:	.BYTE 1		;Wert von AD-Wandler fuer Sensor1
summeadwertL:	.BYTE 1
summeadwertH:	.BYTE 1		;Summe ueber letzte N Werte
nmittel:	.BYTE 1		;N (maximal==64)
ringpuffer:	.BYTE 128	;Platz fuer 64 2-Byte-Werte
ringpufferende:
ringrL:		.BYTE 1		;Lesezeiger auf Ringpuffer
ringrH:		.BYTE 1		;Lesezeiger auf Ringpuffer
ringwL:		.BYTE 1		;Schreibzeiger auf Ringpuffer
ringwH:		.BYTE 1		;Schreibzeiger auf Ringpuffer
adcsummeL:	.BYTE 1		;nur in adcomplete gebraucht
adcsummeH:	.BYTE 1		;nur in adcomplete gebraucht
flashpuffer:	.BYTE PAGESIZEB	;Puffer zum SPM ueberpruefen
zpufferL:	.BYTE 1		;Zeiger auf den Puffer
zpufferH:	.BYTE 1		;Zeiger auf den Puffer
adrunflag:	.BYTE 1		;zum AD-Wandler ein/aus-schalten
glcd_datenA:	.BYTE 20	;Datenbereich fuer glcd.asm ...
glcd_datenB:			;Endmarkierung
grafikposx:	.BYTE 1		;fuer grafische Darstellung
grafikposy:	.BYTE 1		;fuer grafische Darstellung
skalfaktor:	.BYTE 1		;Skalierfaktor fuer Grafik
markposx:	.BYTE 1		;Marker-Position
markposy:	.BYTE 1		;Marker-Position
schuettmodus:	.BYTE 1		;SekSchuettler-Modus
schuettsoll:	.BYTE 1		;SekSchuettler-Sollzeit
.ifdef UARTREADOUT
letztes_rx:	.BYTE 1
sendstatus1:	.BYTE 1	 ;was gesendet werden soll: 'e' fuer EEPROM 'f' fuer FLASH, 'v' fuer Version
sendstatus2:	.BYTE 1	 ;':'=n, 1=1.Adress-Byte, 2=2.Adress-Byte, 3=0|1, 4=Daten, 5=Pruefsumme
send_anzahl:	.BYTE 1	 ;Anzahl Bytes die in aktueller Zeile gesendet werden sollen
send_adrH:	.BYTE 1
send_adrL:	.BYTE 1
send_endeH:	.BYTE 1
send_endeL:	.BYTE 1
send_buf1:	.BYTE 1
pruefsumme:	.BYTE 1
npunkteL:	.BYTE 1
npunkteH:	.BYTE 1		; Anzahl bisher gemessene Punkte
schlussflag:	.BYTE 1		;funktioniert noch nicht
.endif
;neu:		.BYTE 1		;ev. neues hier eintragen!!

errorflag:	.BYTE 1		;zweitletztes im SRAM benutztes Byte
errorflag2:	.BYTE 1		;letztes im SRAM benutztes Byte
;falls errorflag dann vom Stack ueberschriben wird --> Fehlerabbruch
		.CSEG
;-----------------------------------------------------------------------------

.ifdef UARTREADOUT
.equ ZUSATZBLOCK=0x200	;Zusaetzlicher Programmspeicher vom Datenbereich abzwacken
.org LARGEBOOTSTART-ZUSATZBLOCK

.equ BAUD  = 38400                              ; Baudrate
.equ UBRR_VAL   = ((F_CPU+BAUD*8)/(BAUD*16)-1)  ; clever runden
.equ BAUD_REAL  = (F_CPU/(16*(UBRR_VAL+1)))     ; Reale Baudrate
.equ BAUD_ERROR = ((BAUD_REAL*1000)/BAUD-1000)  ; Fehler in Promille
.if ((BAUD_ERROR>10) || (BAUD_ERROR<-10))       ; max. +/-10 Promille Fehler
  .error "Systematischer Fehler der Baudrate grsser 1 Prozent und damit zu hoch!"
.endif

.equ STARTZEICHEN = ':'
.equ STATUS2_ADR1_SENDEN  = 1
.equ STATUS2_ADR2_SENDEN  = 2
.equ STATUS2_0ODER1_SENDEN = 3
.equ STATUS2_DATEN_SENDEN = 4
.equ STATUS2_PRUEFSUMME   = 5
	
rx_complete:			; es wurde ein Zeichen vom UART empfangen
	push r0
	in r0, SREG
	push r16
	push r17
	push r18
	in r16, UDR
	cpi r16, 13		; '\r'
	breq rx_ret		; CR ignorieren, nur LF als Zeilentrenner relevant
	lds r18, letztes_rx
	sts letztes_rx, r16
	cpi r16, 10		; '\n'
	brne rx_ret
	cpi r18, 'f'		; soll FLASH gelesen werden?
	breq flashsenden
	cpi r18, 'e'		; soll EEPROM gelesen werden?
	breq eepromsenden
	cpi r18, 's'		; soll gestoppt werden?
	breq rx_stoppen
.ifdef VERSIONSANZEIGE
	cpi r18, 'v'		; Version senden?
	brne rx_s1
	ldi r16, high(ScattVers<<1) ;ja: String im Flash zur Versionsanzeige
	ldi r17, low(ScattVers<<1)
	sts send_adrH, r16
	sts send_adrL, r17
	sts sendstatus1, r18
	rjmp rx_nextline
.endif
rx_s1:	lds r17, sendstatus1
	tst r17
	breq fehler
	cpi r18, '0'		; letzte Zeile wurde erfolgreich empfangen
	breq rx_nextline
	cpi r18, '1'		; letzte Zeile wurde mit Fehlern empfangen
	breq rx_nochmals
fehler:		; provisorisch: keine Reaktion auf unbekannte Kommandos
rx_ret:	rjmp rx_return
rx_stoppen:
	clr r16
	sts sendstatus1, r16
	sts sendstatus2, r16
	sts send_buf1, r16
	rjmp rx_return
eepromsenden:
	ldi r16, high(EEPROMEND+1)
	ldi r17, low(EEPROMEND+1)
	rjmp sendparametersetzen
flashsenden:
;	ldi r16, high((FLASHEND+1)*2)
;	ldi r17, low((FLASHEND+1)*2)		;gesamtes Flash lesen
	ldi r16, high((LARGEBOOTSTART-ZUSATZBLOCK)*2)
	ldi r17, low((LARGEBOOTSTART-ZUSATZBLOCK)*2)	;nur Daten-Teil vom Flash lesen
	sts npunkteL, r24
	sts npunkteH, r25	; Anzahl bisher gemessene Punkte
	cpi r24, 0		;r25:r24 == 0 ?
	cpc r25, r24
	brne sendparametersetzen
	sts schlussflag, r18	;ja: Flag zum erkennen dass abgeschlossene Messung gelesen wird
		;; Am Schluss wenn letzte Zeile gelesen und Antwort ok ('0') war, muesste es
		;; noch auf 2 gesetzt werden. Aber wie ohne grossen Aufwand?
sendparametersetzen:
	sts sendstatus1, r18
	sts send_endeH, r16
	sts send_endeL, r17
	clr r16
	sts send_adrH, r16	; bei Adresse 0 beginnen
	sts send_adrL, r16
	rjmp rx_nextline
rx_nochmals:			; gleiche Zeile nochmals senden
	lds r16, send_adrL
	lds r17, send_adrH
	subi r16, 16		; letzte 16 Bytes nochmals senden
	sbci r17, 0
	sts send_adrL, r16
	sts send_adrH, r17
rx_nextline:
	clr r16
	sts pruefsumme, r16
	ldi r16, 16		; 16 Bytes pro Zeile senden
	sts send_anzahl, r16
	ldi r16, STARTZEICHEN
	sts sendstatus2, r16
	out UDR, r16		; STARTZEICHEN senden
rx_return:
	pop r18
	pop r17
	pop r16
	out SREG, r0
	pop r0
	reti

tx_complete:			; letztes Zeichen ist jetzt fertig gesendet
	push r0
	in r0, SREG
	push r16
	push r17
	push r18
	push r19
	push r20
	push ZL
	push ZH
	lds ZH, send_adrH
	lds ZL, send_adrL
	lds r18, send_endeL
	lds r19, send_endeH
	lds r16, send_buf1	; noch ein Zeichen im Sendepuffer?
	tst r16
	breq tx_s1
	out UDR, r16		; ja--> dieses senden
	clr r16
	sts send_buf1, r16
	rjmp tx_return
tx_s1:
	lds r17, sendstatus2
	tst r17			; sendstatus leer?
	breq tx_ret		; ja --> nichts machen
	lds r20, sendstatus1
	cpi r20, 'v'		; Versionsanzeige?
	brne tx_s2
tx_version:			; Version als String im Flash senden
	lpm r16, Z+
	sts send_adrH, ZH	; neue aktuelle Adresse speichern
	sts send_adrL, ZL
	tst r16			; String-Ende erreicht?
	brne tx_v1
	sts sendstatus1, r16	; ja: Sendung stoppen
	sts sendstatus2, r16
	ldi r16, 10		; und als Abschluss noch ein '\n' senden
tx_v1:	out UDR, r16
tx_ret:	rjmp tx_return
tx_s2:	cpi r17, STARTZEICHEN
	brne tx_L1
	ldi r17, STATUS2_ADR1_SENDEN-1	; status fuer naechsts Byte vorbereiten
	lds r16, send_anzahl		; dieses Byte als Hexzahl senden
	rjmp tx_inc_bytesenden
tx_L1:	cpi r17, STATUS2_ADR1_SENDEN
	brne tx_L2
	mov r16, ZH
	rjmp tx_inc_bytesenden
tx_L2:	cpi r17, STATUS2_ADR2_SENDEN
	brne tx_L3
	mov r16, ZL
	rjmp tx_inc_bytesenden
tx_L3:	cpi r17, STATUS2_0ODER1_SENDEN
	brne tx_L4
	lds r16, send_anzahl
	sub r18, r16
	sbci r19, 0		; r19:r18 = Endadresse - N
	cp  ZL, r18
	cpc ZH, r19		; wird Ende erreicht?
	ldi r16, 0		; nein:	00 senden
	brcs tx_inc_bytesenden
	ldi r16, 1		; ja: 01 senden
	rjmp tx_inc_bytesenden
tx_L4:	cpi r17, STATUS2_DATEN_SENDEN
	brne tx_L5
	lds r16, send_anzahl
	dec r16
	sts send_anzahl, r16
	brne tx_datensenden
	ldi r17, STATUS2_PRUEFSUMME
	sts sendstatus2, r17
	rjmp tx_datensenden
tx_L5:	cpi r17, STATUS2_PRUEFSUMME
	brne tx_L6
	lds r16, pruefsumme
	rjmp tx_inc_bytesenden
tx_L6:	ldi r16, 10		; '\n' Zeilentrenner senden
	out UDR, r16
	clr r16
	sts sendstatus2, r16
	cp  ZL, r18
	cpc ZH, r19		; Endadresse erreicht?
	brcs tx_return
	sts sendstatus1, r16	; ja: Sendung stoppen
tx_return:
	pop ZH
	pop ZL
	pop r20
	pop r19
	rjmp rx_return
tx_datensenden:
	cpi r20, 'f'		; sendstatus1
	breq tx_flash
;	cpi r20, 'e'
;	breq tx_eeprom
tx_eeprom:
	sbic EECR, EEWE
	rjmp tx_eeprom		; ev. warten bis letzter Schreibzugriff beeendet
	out EEARH, ZH
	out EEARL, ZL
	sbi EECR, EERE		; Lesevorgang aktivieren
	in r16,	EEDR
	adiw ZH:ZL, 1
	rjmp tx_sts
tx_flash:
	cpi r24, 0		; r25:r24 == 0 ist, dann ist noch keine neue Messung gestartet
	cpc r25, r24		; ist schon eine neue Messung gestartet?
	breq tx_f4		; nein--> alte Daten vom Flash lesen
	cp ZH, YH		; Position wo Anzahl Messpunkte hin kommen sollen?
	breq tx_evnpunkte	; eventuell-->
tx_f3:				; nein:
	rcall zxvergleich	; in Bereich wo Puffer noch nicht ins Flash geschrieben wurde ?
	breq tx_vonpuffer	; ja--> vom Puffer lesen
tx_f4:	lpm r16, Z+		; nein:	wirklich vom Flash lesen
tx_sts:	sts send_adrH, ZH	; aktuelle Adresse speichern
	sts send_adrL, ZL
	rjmp tx_bytesenden
tx_inc_bytesenden:
	inc r17
	sts sendstatus2, r17
tx_bytesenden:
	mov r17, r16
	swap r16		; obere 4 Bits
	rcall hexdigit		; in Hex-Digit umwandeln
	out UDR, r16		; und senden.
	lds r16, pruefsumme
	sub r16, r17		; Pruefsumme aktualisieren
	sts pruefsumme, r16
	mov r16, r17
	rcall hexdigit
	sts send_buf1, r16	; untere 4 Bits naechstes Mal senden
	rjmp tx_return
tx_evnpunkte:
	cp ZL, YL		; 1.Position wo Anzahl Messpunkte hin kommen sollen?
	brne tx_f1		; nein-->
	lds r16, npunkteH	; ja: bisherige Anzahl Punkte nehmen
	rjmp tx_f2
tx_f1:	mov r16, YL
	inc r16
	cp  ZL, r16		; 2.Position wo Anzahl Messpunkte hin kommen sollen?
	brne tx_f3		; nein-->
	lds r16, npunkteL	; ja: bisherige Anzahl Punkte nehmen
tx_f2:	adiw ZH:ZL, 1
	rjmp tx_sts
tx_vonpuffer:
	push ZH
	push ZL
	andi ZL, low(PAGESIZEB-1)
	andi ZH, high(PAGESIZEB-1)
	ldi r16, low(flashpuffer)
	ldi r17, high(flashpuffer)
	add ZL, r16
	adc ZH, r17
	ld r16, Z
	pop ZL
	pop ZH
	rjmp tx_f2

zxvergleich:
	push ZH
	push ZL
	push XH
	push XL
	andi ZL, low(~(PAGESIZEB-1))
	andi ZH, high(~(PAGESIZEB-1))
	andi XL, low(~(PAGESIZEB-1))
	andi XH, high(~(PAGESIZEB-1))
	cp  ZL, XL
	cpc ZH, XH
	pop XL
	pop XH
	pop ZL
	pop ZH
	ret

hexdigit:
	andi r16, 0x0F
	ori r16, 0x30
	cpi r16, 0x3A		; Ziffern 0 bis 9 ?
	brcs hex_L1
	subi r16, -7		; nein: 7 addieren fuer A bis F
hex_L1:	ret

.ifdef LEDSCHALTUHR
.def stellmodus = r20
.equ SETLEDON = 1
.equ SETLEDOFF = 2

uvled_init:
	ldi r16, 10		; 10 Minuten Bestrahlung
	sts ledeinzeit, r16
	ldi r16, 20		; 20 Minuten dunkel
	sts ledauszeit, r16
	sts ledschaltzeit, r16	; mit Dunkel-Periode beginnen
	ldi stellmodus, 0
uvled_off:
	sbi PORTB, 5		; UV-LED ausschalten
	ret

uvled_stellen:		; mit Taster an PB6 und PB7, Ruckgabewert: Zero-Flag wenn nicht am Stellen
	cpi stellmodus, SETLEDON
	breq uvst10
	cpi stellmodus, SETLEDOFF
	breq uvst20
	sbic TAST_PIN2, TASTER2	;Skip if Bit Clear, Taster2 gedrueckt?
	rjmp uvst2		;nein-->
uvst1:	rcall milliwait5
	sbis TAST_PIN2, TASTER2	;Skip if Bit Set, Taster2 wieder losgelassen?
	rjmp uvst1		;nein--> warten
	ldi stellmodus, SETLEDON
	rjmp uvled_lcdret
uvst2:	sbic TAST_PIN2, TASTER3	;Skip if Bit Clear, Taster3 gedrueckt?
	rjmp uvled_ret		;nein-->
uvst3:	rcall milliwait5
	sbis TAST_PIN2, TASTER3	;Skip if Bit Set, Taster3 wieder losgelassen?
	rjmp uvst3		;nein--> warten
	ldi stellmodus, SETLEDOFF
	rjmp uvled_lcdret
uvst10:	sbis TAST_PIN2, TASTER2	;Skip if Bit Set, Taster2 gedrueckt?
	rcall incledontime	;ja: Ein-Zeit erhoehen
	sbis TAST_PIN2, TASTER3	;Skip if Bit Set, Taster3 gedrueckt?
	rcall decledontime	;ja: Ein-Zeit erniedrigen
	rjmp uvst30
uvst20:	sbis TAST_PIN2, TASTER2	;Skip if Bit Set, Taster2 gedrueckt?
	rcall incledofftime	;ja: Aus-Zeit erhoehen
	sbis TAST_PIN2, TASTER3	;Skip if Bit Set, Taster3 gedrueckt?
	rcall decledofftime	;ja: Aus-Zeit erniedrigen
uvst30:	rcall milliwait255
	sbic TAST_PIN, TASTER	;Skip if Bit Clr, Taster1 gedrueckt?
	rjmp uvled_lcdret	;nein-->
	ldi stellmodus, 0	;ja: Stellmodus zuruecksetzen
uvst31:	rcall milliwait5
	sbis TAST_PIN, TASTER	;Skip if Bit Set, Taster1 wieder losgelassen?
	rjmp uvst31		;nein--> warten
uvled_lcdret:
	rcall lcd_ledonofftime
uvled_ret:
	tst stellmodus
	ret
incledontime:
	lds r16, ledeinzeit
	inc r16
	breq uvst11
	sts ledeinzeit, r16
uvst11:	ret
decledontime:
	lds r16, ledeinzeit
	dec r16
	cpi r16, 0xFF
	breq uvst12
	sts ledeinzeit, r16
uvst12:	ret
incledofftime:
	lds r16, ledauszeit
	inc r16
	breq uvst21
	sts ledauszeit, r16
uvst21:	ret
decledofftime:
	lds r16, ledauszeit
	dec r16
	cpi r16, 0xFF
	breq uvst22
	sts ledauszeit, r16
uvst22:	ret

;; Fuer Anzeige auf LCD: 'UVON=199 OFF=199'
uvontxt:	.db "UVON:",0
uvofftxt:	.db " OFF:",0
lcd_ledonofftime:
	push r17
	push ZL
	push ZH
	ldi r16, 0x21		; 2.Zeile 1.Zeichen
	rcall lcd_go
	ldi ZL, low(uvontxt<<1)
	ldi ZH, high(uvontxt<<1)
	rcall print_string
	lds r16, ledeinzeit
	rcall lcd_r16dez
;	ldi ZL, low(uvofftxt<<1)
;	ldi ZH, high(uvofftxt<<1)
	rcall print_string
	lds r16, ledauszeit
	rcall lcd_r16dez
	ldi r16, ' '
	rcall lcd_write
	pop ZH
	pop ZL
	pop r17
	ret

uvled_lcdclear:		;entsprechende Anzeigezeile loeschen
	push r16
	push r17
	push r18
	ldi r16, 0x21		; 2.Zeile 1.Zeichen
	rcall lcd_go
	ldi r18, 16
uvclr:	ldi r16, ' '
	rcall lcd_write
	dec r18
	brne uvclr
	lds r16, ledauszeit
	sts ledschaltzeit, r16	; mit Dunkel-Periode beginnen
	pop r18
	pop r17
	pop r16
	ret

uvled_schalten:	
	push r16
	lds r16, ledschaltzeit
	tst r16
	breq led_ret
	dec r16
	sts ledschaltzeit, r16
	brne led_ret
	in r16, PORTB
	andi r16, (1<<5)	; PB5 gesetzt?
	breq led_aus
led_ein:cbi PORTB, 5		; UV-LED einschalten
	lds r16, ledeinzeit
	rjmp led_r1
led_aus:sbi PORTB, 5		; UV-LED ausschalten
	lds r16, ledauszeit
led_r1:	sts ledschaltzeit, r16
led_ret:pop r16
	ret

.endif	;LEDSCHALTUHR

.endif	;UARTREADOUT
	
;-----------------------------------------------------------------------------
.org LARGEBOOTSTART

;-----------------------------------------------------------------------------
; Reset und Interruptvectoren	;VNr. Beschreibung
.ifdef ATMEGA8
begin:	rjmp	main		; 1   Power On Reset
	reti			; 2   Int0-Interrupt
	reti			; 3   Int1-Interrupt
	reti			; 4   TC2 Compare Match
	reti			; 5   TC2 Overflow
	reti			; 6   TC1 Capture
	reti			; 7   TC1 Compare Match A
	reti			; 8   TC1 Compare Match B
	reti			; 9   TC1 Overflow
	rjmp tc0over		; 10  TC0 Overflow
	reti			; 11  SPI, STC = Serial Transfer Complete
	rjmp rx_complete	; 12  UART Rx Complete
	reti			; 13  UART Data Register Empty
	rjmp tx_complete	; 14  UART Tx Complete
	rjmp adcomplete		; 15  ADC Conversion Complete
	reti			; 16  EEPROM ready
	reti			; 17  Analog Comperator
	reti			; 18  TWI (I2C) Serial Interface
	reti			; 19  Store Program Memory ready
.endif
.ifdef ATMEGA328P
begin:	jmp	main		; 1   Power On Reset
	jmp leererint		; 2   Int0-Interrupt
	jmp leererint		; 3   Int1-Interrupt
	jmp leererint		; 4   PCINT0 Pin Change Interrupt
	jmp leererint		; 5   PCINT1 Pin Change Interrupt
	jmp leererint		; 6   PCINT2 Pin Change Interrupt
	jmp leererint		; 7   WDT Watchdog Iime-out
	jmp leererint		; 8   TIMER2 COMPA Compare Match A
	jmp leererint		; 9   TIMER2 COMPB Compare Match B
	jmp leererint		; 10  TIMER2 OVF Overflow
	jmp leererint		; 11  TC1 Capture
	jmp leererint		; 12  TC1 Compare Match A
	jmp leererint		; 13  TC1 Compare Match B
	jmp leererint		; 14  TC1 Overflow
	jmp leererint		; 15  TC0 Compare Match A
	jmp leererint		; 16  TC0 Compare Match B
	jmp tc0over		; 17  TC0 Overflow
	jmp leererint		; 18  SPI, STC = Serial Transfer Complete
	jmp rx_complete		; 19  UART Rx Complete
	jmp leererint		; 20  UART Data Register Empty
	jmp tx_complete		; 21  UART Tx Complete
	jmp adcomplete		; 22  ADC Conversion Complete
	jmp leererint		; 23  EEPROM ready
	jmp leererint		; 24  Analog Comperator
	jmp leererint		; 25  TWI (I2C) Serial Interface
	jmp leererint		; 26  SPM READY Store Program Memory ready
leererint:
	reti
.endif
.ifdef ATMEGA32
begin:	jmp	main		; 1   Power On Reset
	jmp leererint		; 2   Int0-Interrupt
	jmp leererint		; 3   Int1-Interrupt
	jmp leererint		; 4   Int2-Interrupt
	jmp leererint		; 5   TC2 (Timer/Counter2) Compare Match
	jmp leererint		; 6   TC2 OVF Overflow
	jmp leererint		; 7   TC1 Capture
	jmp leererint		; 8   TC1 Compare Match A
	jmp leererint		; 9   TC1 Compare Match B
	jmp leererint		; 10  TC1 Overflow
	jmp leererint		; 11  TC0 Compare Match
	jmp tc0over		; 12  TC0 Overflow
	jmp leererint		; 13  SPI, STC = Serial Transfer Complete
	jmp rx_complete		; 14  USART Rx Complete
	jmp leererint		; 15  USART Data Register Empty
	jmp tx_complete		; 16  USART Tx Complete
	jmp adcomplete		; 17  ADC Conversion Complete
	jmp leererint		; 18  EEPROM ready
	jmp leererint		; 19  Analog Comperator
	jmp leererint		; 20  TWI (I2C) Two-wire Serial Interface
	jmp leererint		; 21  SPM_RDY Store Program Memory ready
leererint:
	reti
.endif
.ifndef UARTREADOUT
rx_complete:	
tx_complete:	reti
.endif

;-----------------------------------------------------------------------------
; Start, Power On, Reset
main:	ldi r16, RAMEND&0xFF
        out SPL, r16  	     ; Init Stackpointer L
	ldi r16, RAMEND>>8
        out SPH, r16  	     ; Init Stackpointer H

;; Init-Code
	ldi r16, 0x37		;Leitungen fuer LEDs und PB4, PB5 auf Ausgang
        out DDRB, r16		;Data Direction Register B setzen
	ldi r16, 0x20		;0x20=UVLED aus, positive Schuettler-Logik: +0x10
	out PORTB, r16		;alle LEDs loeschen, Schuettler in Aus-Position
	sbi TAST_PORT, TASTER	;Pullup-Widerstand fuer Taster
.ifdef LEDSCHALTUHR
	sbi TAST_PORT2, TASTER2	;Pullup-Widerstand fuer Taster2
	sbi TAST_PORT2, TASTER3	;Pullup-Widerstand fuer Taster3
.endif
.ifdef GRAFIK_LCD
	rcall glcd_init1
	ldi r17, 1
	sts skalfaktor, r17
.endif

	clr r16
.ifdef MARKER
	sts markposy, r16
.endif
.ifdef SEKSCHUETTLER
	sts schuettmodus, r16
.endif

.ifdef ATMEGA328P
;; Timer ist auf dem ATmega328P anders als beim ATmeag8:
	ldi r17, 0x01
	sts TIMSK0, r17		;Timer/Counter0 Overflow einschalten
	ldi r17, 3		;1=nicht teilen, 2=durch 8, 3=64, 4=256, 5=1024
	out TCCR0B, r17		;Vorteiler auf 64 setzen
.else
;; beim ATmega32 gleich wie beim ATmeag8:
	ldi r17, 0x01
	out TIMSK, r17		;Timer/Counter0 Overflow einschalten
	ldi r17, 3		;1=nicht teilen, 2=durch 8, 3=64, 4=256, 5=1024
	out TCCR0, r17		;Vorteiler auf 64 setzen
.endif

;Timer-Einstellungen:
;Schrittweite = 1/Frequenz * Vorteiler (Freqeunz=Hz, Zeit in usec)
;Anzahl Schritte fuer 1msec:  n = 1000usec/Schrittweite
;und erste Formel eingesetzt: n = 1000usec*Frequenz/Vorteiler
;oder mit Frequenz in Hz: n = Frequenz/Vorteiler/1000
;Fuer 8MHz also n=125, bei 16MHz n=250, (bei 3.6864MHz n=57.6)
;Da der Zaehler beim Ueberlauf (256) einen Interrupt ausloest muss jeweils
;bei 256-n gestartet werden: fuer 8MHz 256-125=131
;(bei 3.6864MHz 256-57.6=198.4 oder jeweils 2*199 und 3*198)
.equ ZAEHLER = 256-F_CPU/64/1000
	ldi r17, ZAEHLER
	out TCNT0, r17

.ifdef FLASHVERSION
.ifdef ATMEGA328P
	ldi r17, (1<<IVCE)	;Veraendern von IVSEL vorankuendigen
	out MCUCR, r17		; MCUCR beim 328P
	ldi r17, (1<<IVSEL)	;Interrupttabelle auch bei Bootadresse
	out MCUCR, r17
.else
	ldi r17, (1<<IVCE)	;Veraendern von IVSEL vorankuendigen
	out GICR, r17		; GICR bei atmega8 und atmega32
	ldi r17, (1<<IVSEL)	;Interrupttabelle auch bei Bootadresse
	out GICR, r17
.endif
.endif
	clr r9			; Flags loeschen
	clr r10
	clr r11			; r11r10 = Millisec
	clr r12			; Sekunden
	clr r13			; Minuten
	clr r14			; Stunden
	clr r15			; Tage, auf 0 setzen
	sts errorflag, r15	; errorflag auf 0 setzen
	sts errorflag2, r15	; errorflag2 auf 0 setzen

.ifdef UARTREADOUT
	clr r16
	sts letztes_rx, r16
	sts sendstatus1, r16
	sts sendstatus2, r16
	sts send_buf1, r16
	sts schlussflag, r16
	ldi r16, HIGH(UBRR_VAL)
	out UBRRH, r16
	ldi r16, LOW(UBRR_VAL)
	out UBRRL, r16
	ldi r16, (1<<RXEN) | (1<<TXEN) | (1<<RXCIE) | (1<<TXCIE) ;Senden und Empfangen mit Interrupts
.ifdef ATMEGA328P
	out UCSR0B, r16
	ldi r16, (1<<UCSZ01)|(1<<UCSZ00) 	; fuer ATmega328P
	out UCSR0C, r16
.else
	out UCSRB, r16
	ldi r16, (1<<URSEL)|(3<<UCSZ0)	;Asynchron 8N1 Atmega32 oder Atmega8
	out UCSRC, r16
.endif
.endif

.ifdef LEDSCHALTUHR
	rcall uvled_init
.endif

	sei			;Interrupt erlauben

; AD-Wandler initialisieren:
	ldi r16, 8		;Anzahl Aufsummierungen 
	mov r4, r16		;fuer AD-Wandler
	ldi XH, 0
	ldi XL, 0		;X auf Anfang im EEPROM oder FLASH setzen
;	ldi r16, 0x40		;ADC0 und Referenz=AVCC (+ externem Condenser)
	ldi r16, 0xC0		;ADC0 und Referenz=2.56V, beim 328P ca 1.095V
				;gemessen beim ATmega32: 2.53V
;	ldi r16, 0x00		;ADC0 und Referenz=extern, 2.6V mit Spannungs-
				;teiler 2.2k, 2.0k realisiert.
.ifdef ATMEGA328P
	sts ADMUX, r16
.else
	out ADMUX, r16
.endif
;Prescaler: 5=32 6=64 7=128, 8MHz/64=125kHz (muss im Bereich 50-200kHz liegen)
;AD-Wandlung braucht jeweils 13 Takte also 104usec
;.equ PRESCALER = int(log(F_CPU/200000)/log(2))
.equ	PRESCALER=6  ;fuer 8MHz
;.equ	PRESCALER=7  ;fuer 16MHz
;r16 = Prescaler+0b11001000 (ADEN+ADSC+ADIE = Enable,Einzelstart,Interrupt)
;ADIF (0x10) wird automatisch vom AD-Wandler gesetzt wenn Wandlung fertig.
	ldi r16, (1<<ADEN)|(1<<ADSC)|(1<<ADIE) + PRESCALER
.ifdef ATMEGA328P
	sts ADCSRA, r16		;Prescaler und Enable
.else
	out ADCSRA, r16		;Prescaler und Enable
.endif
		;ab hier wird also der AD-Wandler dauernd laufen
	clr r1			;Start-Parameter
	inc r1			;Start mit Roter LED
	mov r3, r1		;r3=1
	mov r2, r1		;r2=1 -> mit Mittelung (0=ohne)

.ifdef EEPROMVERSION
	.equ MAX=512		;Atmega8 hat 512 Bytes = 256 WORDs EEPROM
.endif

.ifdef FLASHVERSION
  .ifdef ZUSATZBLOCK
	.equ MAX=(LARGEBOOTSTART-ZUSATZBLOCK)*2
  .else
	.equ MAX=LARGEBOOTSTART*2
  .endif
.endif

.if DATENFORMAT==0
	ldi r22, low(MAX-6)	;Freier Speicher minus 6 Bytes Kopfdaten 
	ldi r23, high(MAX-6)
.else ;if DATENFORMAT<=2
	ldi r22, low(MAX-8)	;bei gepacktem Speichern noch 2 Reserve
				;zum am Schluss noch ev. letztes Nibble zu speichern
				;(2 weil manchmal gleich 2 Byte gespeichert werden)
	ldi r23, high(MAX-8)
.endif
	clr r24
	clr r25			;Anzahl Messpunkte = 0

	clr r17			;r17=0
	ldi r18, SENSORSPEICHERN
	clr r5			;Nibble fuer gepacktes Speichern loeschen
.ifdef LCDANZEIGE
	rcall milliwait255
	rcall lcd_init
.endif

;----------------------------- tests -----------------------------------------
.ifdef GLCDTEST
	push r16
	push r17
	push r18
	ldi r16, 1  		;1.Zeile 3.Zeichen
	ldi r17, 3
	rcall lcd_goto
	ldi r16, 'A'
	rcall glcd_writec
	ldi r16, 'B'
	rcall glcd_writec
	ldi r16, 'C'
	rcall glcd_writec
	ldi r16, 2		; 2.Zeile 1.Zeichen
	ldi r17, 1
	rcall lcd_goto
	ldi r16, 'D'
	rcall glcd_writec
	ldi r16, 3		; 3.Zeile 1.Zeichen
	ldi r17, 1
	rcall lcd_goto
	ldi r16, 'E'
	rcall glcd_writec
	ldi r16, 4		; 4.Zeile 2.Zeichen
	ldi r17, 2
	rcall lcd_goto
	ldi r16, 'F'
	rcall glcd_writec
	ldi r16, 5		; 5.Zeile 3.Zeichen
	ldi r17, 3
	rcall lcd_goto
	ldi r16, 'G'
	rcall glcd_writec
	ldi r16, high(1234)
	ldi r17, low(1234)
	ldi r18, 1
	rcall lcd_r16r17
	inc r17
	rcall lcd_r16r17
	inc r17
	rcall lcd_r16r17
	inc r17
	rcall lcd_r16r17
	pop r18
	pop r17
	pop r16
.endif
.ifdef GRAFIKTEST
	push r2
	push r3
	push r16
	push r17
	push r18
	push r19
	push r20
	ldi r16, low(511)
	ldi r17, high(511)	;Start der Linie in der Mitte linker Kante
	ldi r19, 128		;Breite des Displays in Punkten
gtest1:	mov r2, r16
	mov r3, r17
	ldi r18, 1
	rcall punktgrafischdarstellen
	mov r16, r2
	mov r17, r3
	subi r16, -4		; 4 addieren
	sbci r17, -1
	dec r19
	brne gtest1		;Linie bis rechte obere Ecke zeichnen
	pop r20
	pop r19
	pop r18
	pop r17
	pop r16
	pop r3
	pop r2
.endif
.ifdef TESTLOOP
testloop1:			; test
	in r16,	PORTB
	andi r16, 0xF8		; nur untere 3 Bits
	mov r17, r12		; r12=sec
	andi r17, 7		; untere 3 Bits von sec
	or r16, r17
	out PORTB, r16		; test: Sekunden auf LEDs
	rjmp testloop1		; test
.endif
;----------------------------- tests -----------------------------------------

;---------------------------- Hauptprogramm ----------------------------------
.ifdef LEDSCHALTUHR
;	rcall lcd_ledonofftime
.endif
startloop:
.ifdef LEDSCHALTUHR
	clr r12			;Sekunden=0, Uhr erst wirklich laufen lassen wenn Messung startet
	rcall uvled_stellen
	brne startloop
.endif
	rcall ledportb		;ausgewaehlte LED leuchten lassen
.ifdef LCDANZEIGE
	rcall lcd_r1
.endif
	sbic TAST_PIN, TASTER	;Skip if Bit Clear, Taster gedrueckt?
	rjmp startloop		;nein--> warten
	rcall milliwait255	;255ms warten
	sbis TAST_PIN, TASTER	;Skip if Bit Set, Taster losgelassen?
	rjmp evlangertastendruck ;nein--> langer Tastendruck
kurzertastendruck:
	lsl r1			;naechste LED
	sbrs r1, 3		;0x08 erreicht ?
	rjmp startloop		;nein-->
	mov r1, r3		;ja: r1=1
;	eor r2, r3		;r2 zwischen 1 und 0 wechseln, zum Mittelung ausschalten
	rjmp startloop
evlangertastendruck:	
	rcall milliwait255	;255ms warten
	sbic TAST_PIN, TASTER	;Skip if Bit Clear, Taster noch gedrueckt?
	rjmp kurzertastendruck	;nein-> doch nur kurzer Tastendruck
langertastendruck:
	rcall ledportbclear
.ifdef LEDSCHALTUHR
	rcall uvled_lcdclear
.endif
	rcall milliwait255	;255ms warten
	rcall ledportb
	rcall milliwait255	;255ms warten
	sbis TAST_PIN, TASTER	;Skip if Bit Set, Taster losgelassen?
	rjmp langertastendruck	;nein--> warten
aufganzesekundewarten:
	mov r20, r12		;r20 = Startzeit Sekunden
aufsecwarten:
	cp r20, r12
	breq aufsecwarten
	mov r20, r12		;r20 = Startzeit Sekunden
	mov r21, r13		;r21 = Startzeit Minuten

; r1 --> r2 = Messperiode in Anzahl Sekunden:
	sbrc r1, 0		;Skip if Bit in Register Clear, Rot gewaehlt?
	rjmp auswahl1		;ja-->
	sbrc r1, 1		;Skip if Bit in Register Clear, Gruen gewaehlt?
	rjmp auswahl2		;ja-->
auswahl3:
;	ldi r16, 10		;Schaltzeit Magnetruehrer
	ldi r19, 60		;60 Sekunden Messperiode
	rjmp auswahlfertig
auswahl1:
;	ldi r16, 0		;Schaltzeit Magnetruehrer deaktivieren
	ldi r19, 1		;1 Sekunde Messperiode
.ifdef SEKSCHUETTLER
	sts schuettmodus, r19
	sts schuettsoll, r12
.endif
	rjmp auswahlfertig
auswahl2:
;	ldi r16, 2		;Schaltzeit Magnetruehrer
	ldi r19, 10		;10 Sekunden Messperiode
auswahlfertig:
;	sts magnetschaltzeit, r16
	mov r16, r19
	tst r2
	brne ausw3
	ldi r16, 1		;N=1 --> ohne Mittelung
ausw3:	sts nmittel, r16	;N fuer laufender Mittelwert
	rcall ringinit

;Datenkopf schreiben:
	ldi r16, DATENFORMAT
	rcall bytespeichern
	ldi r16, SENSORTYP
	rcall bytespeichern
.if SENSORSPEICHERN==2
	ldi r16, ANSCHLUSSVARIANTE2
.else
	ldi r16, ANSCHLUSSVARIANTE
.endif
	rcall bytespeichern
	mov r16, r19
	rcall bytespeichern	;Messperiode in Sekunden
;	clr r24			;schon weiter oben gemacht
;	clr r25			;Anzahl Messpunkte = 0
	mov YH,XH		;Y = Zeiger zum am Schluss Anzahl
	mov YL,XL		;Messpunkte zu speichern.
	ldi r16, 0xFF
	rcall bytespeichern	;Anzahl Messpunkte erst mal auf ungueltigen
	rcall bytespeichern	;Wert (0xFFFF) setzen.
;	clr r6			;Letzter Messwert auf 0 setzen nicht noetig
;	clr r7			;weil erster Messwert eh Absolutwert.
	rcall admessung		;AD-Wert lesen und ev. aufsummieren -> r16:r17 
messloop:
	rcall messung		;AD-Wert von r16:r17 speichern
	add r20, r19		;neue Sollzeit = alte Sollzeit + Messperiode
	cpi r20, 60		;r20<60 ?
	brlt warteloop		;ja-->
	subi r20, 60		;nein: 60 subtrahieren
	;braucht da noch Anpassung falls Messperiode groesser 60 sein kann!
	inc r21			;und Minuten erhoehen
	cpi r21, 60
	brne warteloop
	clr r21
warteloop:
	mov r17, r12
nextsec:
	sbis TAST_PIN, TASTER	;Skip if Bit Set, Taster gedrueckt?
	rjmp evvorzeitig	;ja--> ev. vorzeitig beenden
	lds r16, errorflag
	tst r16
	brne fehlerabbruch
	lds r16, errorflag2
	tst r16
	brne fehlerabbruch2
	cp r17, r12
	breq nextsec		;auf naechste Sekunde warten
	rcall admessung		;AD-Wert lesen und ev. aufsummieren -> r16:r17
.ifdef MAGNETRUEHRER
	rcall magnetruehrer
.endif
.ifdef SCHUETTLER
	rcall schuetteln
.endif
	cp r20, r12		;Sollzeit Sekunden erreicht?
	cpc r21, r13		;Sollzeit Minuten erreicht?
	brne warteloop
.ifdef SCHUETTLER
.ifdef SEKSCHUETTLER
	rcall ev_adwandler_ein
.else
	rcall adwandler_ein
.endif
.endif
	tst r23			;Speicher voll: restlicher Platz == 0?
	brne messloop		;nein-->
	tst r22
	brne messloop		;nein-->
	rjmp mfertig
fehlerabbruch2:
.ifdef STACKERROR
	rcall print_stack	;Ausgabe "Stack" auf LCD
.endif
fehlerabbruch:
.ifdef LCDANZEIGE
	ldi r16, 0x11
	rcall lcd_go
	rcall print_error	;Ausgabe "Error" auf LCD
.endif
	rjmp halteloop
.ifdef TASTMARKER
tastmark:			;Taster wurde kurz gedrueckt --> Marker setzen
	sbi PORTB, 0		; rote LED an
	cp r17, r12
	breq tastmark		;naechste Sekunde noch abwarten
	ldi r16, 0
	ldi r17, 0		;statt admessung einen Wert von 0 als Marker speichern
	rjmp messloop		;weiter in der Messloop
.endif
evvorzeitig:
	ldi r16, 250
	rcall milliwait1	;250ms warten
	sbis TAST_PIN, TASTER	;Skip if Bit Set, Taster immer noch gedrueckt?
	rcall milliwait1	;ja: nochmals 250ms warten
	sbic TAST_PIN, TASTER	;Taster immer noch gedrueckt?
.ifdef TASTMARKER
	rjmp tastmark
.else
	rjmp nextsec		;nein--> dochnicht
.endif
mfertig:
	sbrs r5, 4		;Bit 4 in r5 gesetzt?
	rjmp mfert2		;nein --> kein vorheriges Nibble
	mov r16, r5
	swap r16
	andi r16, 0xF0		;nicht unbedingt noetig (unteres Nibble wird
				;eh verworfen), aber leichter zum debuggen.
	rcall bytespeichern	;letztes Nibble auch noch speichern
mfert2:
.ifdef EEPROMVERSION
	mov XH, YH
	mov XL, YL
	mov r16, r25
	rcall eepromstore
	mov r16, r24
	rcall eepromstore	;Anzahl Messpunkte speichern
.else
	ldi r16, 0xFF
	rcall endeimflashspeichern	;Rest der Page mit 0xFF fuellen
; Erste Page wurde schon in imflashspeichern ins SRAM kopiert
	ldi XL, low(SRAMSTART)
	ldi XH, high(SRAMSTART)
	add XL, YL
	adc XH, YH
	st X+, r25		;Anzahl Messpunkte im SRAM
	st X+, r24		;noch richtig eintragen.
	rcall page1zurueckkopieren
.endif
halteloop:
.ifdef UARTREADOUT
	clr r24			;Anzahl Messpunkte = 0 (fuer neue Messung)
	clr r25			;(wichtig fuer tx_flash)
.endif
.ifdef LEDSCHALTUHR
	rcall uvled_off
.endif
	lsl r1
	sbrc r1, 3		;0x08 erreicht ?
	mov r1, r3		;ja: r1=1
	rcall ledportb
	rcall milliwait255	;255ms warten
.ifndef UARTREADOUT
	rjmp halteloop
.else
	lds r16, schlussflag
	cpi r16, 2		; Flash wirklich fertig gelesen? (geht noch nicht)
	brne halteloop
	rcall milliwait255	;255ms warten
	rjmp halteloop		; provisorisch
;	cli			;wenn am Schluss das Flash fertig ausgelesen wurde --> Neustart
;	rjmp main
.endif

ledportb:
	push r16
	in r16, PORTB
	andi r16, 0xF8		;obere Bits unveraendert lassen
	or r16, r1		;untere 3 Bits = ausgewaehlte LED
	rjmp ledp1
ledportbclear:
	push r16
	in r16, PORTB
	andi r16, 0xF8		;obere Bits unveraendert lassen
ledp1:	out PORTB, r16
	pop r16
	ret
	
.ifdef FLASHVERSION

.else
bytespeichern:
eepromstore:
;	st X+, r16	;fuer test mit SRAM
;	ret		;fuer test mit SRAM
eeL1:	sbic EECR,1
	rjmp eeL1	;warte bis EEPROM bereit
	out EEARH,XH
	out EEARL,XL
	out EEDR, r16
	sbi EECR, 2	;EEMWE-Bit setzen
	sbi EECR, 1	;EEWE-Bit setzen
	adiw XL, 1
	ret
.endif

page1zurueckkopieren:		;Page vom SRAM zurueck ins FLASH kopieren
	ldi XL, 0
	ldi XH, 0
	ldi ZL, low(SRAMSTART)
	ldi ZH, high(SRAMSTART)
	ldi r17, PAGESIZEB
kopL2:	ld r16, Z+
	rcall imflashspeichern
	dec r17
	brne kopL2
	ret

;-----------------------------------------------------------------------------
; Interrupt-Routine
; Registerbelegungen:
; r11:r10 Millisekunden, r12 Sekunden, r13 Minuten, r14 Stunden, r15 Tage
tc0over:push r16
	push r17
	in r17, SREG	;Statusregister sichern
	push r17
	ldi r17, ZAEHLER
	out TCNT0, r17
	ldi r16, 0xE8
	ldi r17, 0x03	;r17:r16 = 1000 = 0x03E8
	inc r10		;Millisekunden erhoehen
	brne L1
	inc r11
L1:	cp r10, r16
	cpc r11, r17
	brne end1	;fertig wenn msec!=1000
	clr r10
	clr r11		;Millisekunden auf 0 setzen
	inc r12		;Sekunden erhoehen
	ldi r16, 60
	cp r12, r16
	brne end1	;fertig wenn sec!=60
	clr r12		;Sekunden auf 0 setzen
.ifdef LEDSCHALTUHR
	rcall uvled_schalten
.endif
	inc r13		;Minuten erhoehen
	cp r13, r16
	brne end1	;fertig wenn min!=60
	clr r13		;Minuten auf 0 setzen
	inc r14		;Stunden erhoehen
	ldi r16, 24
	cp r14, r16
	brne end1	;fertig wenn std!=24
	clr r14		;Stunden auf 0 setzen
	inc r15		;Tage erhoehen
end1:
pop17reti:
	pop r17
	out SREG, r17	;Statusregister zurueckholen
	pop r17
	pop r16
	reti

;-----------------------------------------------------------------------------
; Interrupt-Routine fuer AD-Wandler
; Rueckgabewert: sensor1adwert
adcomplete:
	push r16
	push r17
	in r17, SREG	;Statusregister sichern
	push r17
	push r30
	push r31
.ifdef ATMEGA328P
	lds r17, ADCL	;L muss zuerst gelesen werden!
	lds r16, ADCH
.else
	in r17, ADCL	;L muss zuerst gelesen werden!
	in r16, ADCH
.endif
	lds r30, adcsummeL
	lds r31, adcsummeH
	add r30, r17
	adc r31, r16	;Summe += Messwert
	dec r4		;letzter Messert?
	brne _end3	;nein-->
	ldi r17, 3
durch8:	asr r31		;Durchschnitt berechnen:
	ror r30		;Summe /= 8
	dec r17
	brne durch8
_L1:	mov r17, r30
	mov r16, r31
	sts sensor1adwertL, r17 	;Messwert im RAM
	sts sensor1adwertH, r16		;zwischenspeichern.
	clr r30
	clr r31
	ldi r17, 8
	mov r4, r17		;Anzahl Aufsummierungen wieder setzen
_end3:	lds r17, adrunflag
	tst r17
	breq _end4
.ifdef ATMEGA328P
;	lds r17, ADCSRA
;	ori r17, (1<<ADSC)
	ldi r17, 0b11001000 + PRESCALER
	sts ADCSRA, r17	;Starte naechste AD-Wandlung (ATmega328P)
.else
	sbi ADCSRA, 6	;Starte naechste AD-Wandlung (ATmega8, ATmega32)
.endif
_end4:	sts adcsummeL, r30
	sts adcsummeH, r31
	pop r31
	pop r30
	rjmp pop17reti

;adwandler_aus:
;	push r17
;	clr r17
;	sts adrunflag, r17
;	pop r17
;	ret

.ifdef SEKSCHUETTLER
ev_adwandler_ein:
	push r20
	lds r20, schuettmodus
	tst r20			;Schuettelmodus eingestellt?
	breq ev_ein		;nein--> AD einschalten
	lds r20, schuettsoll	;Sollzeit Sekunden setzen
	subi r20, -3		;Ende vom Schuetteln bei Sollzeit+3
	cpi r20, 60
	brcs ev_1
	subi r20, 60
ev_1:	cp r20, r12		;Sollzeit+3 erreicht?
	brne ev_ret		;nein--> AD noch nicht einschalten
ev_ein:	rcall adwandler_ein
ev_ret:	pop r20
	ret
.endif

adwandler_ein:
	push r17
.ifdef ATMEGA328P
;	lds r17, ADCSRA
;	ori r17, (1<<ADSC)
	ldi r17, 0b11001000 + PRESCALER
	sts adrunflag, r17
	sts ADCSRA, r17	;Starte naechste AD-Wandlung (ATmega328P)
.else
	ldi r17, 1
	sts adrunflag, r17
	sbi ADCSRA, 6	;Starte naechste AD-Wandlung (ATmega8, ATmega32)
.endif
	pop r17
	ret

;-----------------------------------------------------------------------------
; Routinen fuer Ring-Puffer (FIFO-Speicher)
; mit ringpush wird hineingeschrieben, mit ringpop wieder zurueckgeholt
ringinit:			; Ringpuffer initialisieren, r16 zerstoert
	push XH
	push XL
	push r17
	push r18
	push r2
	push r3
	ldi XL, low(ringpuffer)
	ldi XH, high(ringpuffer)
	sts ringrL, XL
	sts ringrH, XH
	sts ringwL, XL
	sts ringwH, XH		; wenn ringr==ringw ist Puffer leer
	lds r16, sensor1adwertH
	lds r17, sensor1adwertL
	clr r2
	clr r3			; summe=0
	lds r18, nmittel
	tst r18			; sicherheithalber auf 0 testen
	breq ringi2		; N==0 --> keine Anfangswerte speichern
ringi1:	rcall ringpush		; N mal Anfangs-AD-Wert speichern
	add r2, r17
	adc r3, r16		; summe += Startwert
	dec r18
	brne ringi1
ringi2:	sts summeadwertL, r2
	sts summeadwertH, r3
	pop r3
	pop r2
	pop r18
	pop r17
	rjmp popxret
ringzeigercheck:	;XH:XL auf ringpufferende pruefen, und wenn erreicht..
	cpi XL, low(ringpufferende)
	brne ring1
	cpi XH, high(ringpufferende)
	brne ring1
	ldi XL, low(ringpuffer)	 ; ..wieder auf Ringanfang setzen.
	ldi XH, high(ringpuffer)
ring1:	ret
ringpush:			; r16:r17 in den Ringpuffer schreiben
	push XH
	push XL
	lds XL, ringwL
	lds XH, ringwH
	st X+, r16
	st X+, r17
	rcall ringzeigercheck
	sts ringwL, XL
	sts ringwH, XH
	rjmp popxret
ringpop:			; r16:r17 aus dem Ringpuffer auslesen
	push XH
	push XL
	lds XL, ringrL
	lds XH, ringrH
	ld r16, X+
	ld r17, X+
	rcall ringzeigercheck
	sts ringrL, XL
	sts ringrH, XH
popxret:
	pop XL
	pop XH
	ret

;-----------------------------------------------------------------------------
; Subroutine gepacktspeichern(UWORD r16:r17)
; r25:r24 = Anzahl bisherige Messpunkte, unveraendert
; r5 = vorheriges Nibble falls Bit 4 gesetzt, wird aktualisiert
; r7:r6 = alter Messwert, wird dann aktualisiert
; r16:r17 = neuer Messwert, geht verloren, ja wirklich r16=H r17=L
gepacktspeichern:
	push r1
	push r2
	mov r1, r16
	mov r2, r17		;Aktueller Messwert: r1:r2 = r16:r17
	tst r24			;Anzahl bisherige Messpunkte == 0 ?
	brne packstart		;nein -->
	tst r25
	breq pack0		;ja --> ungepackter Absolutwert
packstart:
;       1X -> 4Bit, davon 3Bit Differenzwert (-4 ... +3)
;       01 -> 8Bit, davon 6Bit Differenzwert (-36 ... -5 und +4 ... +35)
;       00 -> 12Bit, davon 10Bit Absolutwert (0 ... 1023)
;mit DATENFORMAT==2:
;       00 -> 16Bit, davon 14Bit Absolutwert (0 ... 16383)
	sub r17, r6	;r7:r6 = alter Wert, (r1:r2 = neuer Wert)
	sbc r16, r7	;r16:r17 = Differenz
	tst r16
	breq pack9p	;positive Differenz weniger als 9 Bit -->
	cpi r16, 0xFF	;negative Differenz weniger als 9 Bit ?
	brne pack0	;nein --> ungepackter Absolutwert
pack9:	cpi r17, 0xFC	;passt negativer Wert in 3 Bit?
	brcc pack1	;ja --> in 4 Bit = 1Nibble packen
	subi r17, -4	;Diff.Wertebereich erhoehen
	cpi r17, 0xE0	;passt negativer Wert in 6 Bit?
	brcs pack0	;nein --> Absolutwert speichern
	rjmp pack2	;ja --> in 2 Nibbles packen
pack9p:	cpi r17, 0x04	;passt positiver Wert in 3 Bit?
	brcs pack1	;ja --> in 1Nibble packen
	subi r17, 4	;Diff.Wertebereich erhoehen
	cpi r17, 0x20	;passt positiver Wert in 6 Bit?
	brcc pack0	;nein --> Absolutwert speichern
pack2:			;in 8 Bit = 2Nibble packen
	andi r17,0x3F	;6 Nutzbits
	ori r17, 0x40	;01 als erste 2 Bits
	sbrs r5, 4	;Bit 4 in r5 gesetzt?
	rjmp pack2k	;nein --> kein vorheriges Nibble
	swap r5		;vorheriges Nibble zuerst..
	mov r16, r5
	andi r16, 0xF0	 ;und zusammen mit..
	push r17
	swap r17
	andi r17, 0x0F 	  ;erstem Nibble..
	add r16, r17
	pop r17
	rcall bytespeichern2  ;abspeichern.
	andi r17,0x0F		;zweites Nibble
	ori r17, 0x10		;als gueltig markieren
	mov r5, r17		;und nach r5 kopieren.
	rjmp packret
pack2k:	mov r16, r17		;genau 1 Byte speichern
	rcall bytespeichern2
	rjmp packret
pack1:	;in 4 Bit packen
	andi r17,0x07
	ori r17, 0x08
	sbrs r5, 4		;Bit 4 in r5 gesetzt?
	rjmp pack1k		;nein --> kein vorheriges Nibble
	swap r5			;ja: vorheriges Nibble einfuegen
	mov r16, r5
	andi r16, 0xF0
	add r16, r17
	rcall bytespeichern2
	clr r5			;r5 als ungueltig markieren
	rjmp packret
pack1k:	ori r17, 0x10		;als gueltig markieren
	mov r5, r17		;und in r5 kopieren
packret:mov r7, r1
	mov r6, r2		;letzter Messwert r7:r6 = r1:r2 = neuer Wert
	rjmp packret2
.if DATENFORMAT==2
pack0:				;16Bit Absolutwert (=r1:r2) speichern
	mov r7, r1
	mov r6, r2		;letzter Messwert r7:r6 = r1:r2 = neuer Wert
	sbrs r5, 4		;Bit 4 in r5 gesetzt?
	rjmp pack04		;nein --> kein vorheriges Nibble
	ldi r17, 0x18		;ja: schiebe Nibble r5 -> r1:r2 -> r5
paL2:	lsr r5
	ror r1
	ror r2
	ror r17
	brcc paL2
	swap r17
	mov r5, r17		;uebriges Nibble
pack04:	mov r16, r1
	rcall bytespeichern2 	;vorheriges und oberstes Nibble speichern
	mov r16, r2
	rcall bytespeichern2	;2 mittlere Nibbles speichern
.else
pack0:				;12Bit Absolutwert (=r1:r2) speichern
	sbrs r5, 4		;Bit 4 in r5 gesetzt?
	rjmp pack03		;nein --> kein vorheriges Nibble
	;andi r1, 0x03	;nur noetig wenn obere Bits in r1 nicht schon 0 waeren.
	mov r16, r5		;ja: Nibble r5 -> oberes Nibble r16:r17
	clr r5
	swap r16		;r16 = r5<<4
	andi r16, 0xF0
	add r16, r1
	rcall bytespeichern2 	;2 obere Nibbles zuerst speichern
	mov r16, r2
	rcall bytespeichern2	;2 untere Nibbles danach speichern
	rjmp packret
pack03:				;3 Nibbles speichern
	mov r7, r1
	mov r6, r2		;letzter Messwert r7:r6 = r1:r2 = neuer Wert
	ldi r16, 4
paL1:	lsl r2
	rol r1
	dec r16
	brne paL1		;r1:r2 um 4 Bit nach links schieben
	mov r16, r1
	rcall bytespeichern2	;erste 2 Nibbles wirklich speichern
	mov r5, r2
	swap r5			;drittes Nibble im r5
	set			;T-Flag setzen
	bld r5, 4		;in r5 Bit4=T == r5 als gueltig markieren
.endif
packret2:
	pop r2
	pop r1
	ret

;-----------------------------------------------------------------------------
milliwait5:
	ldi r16, 5
milliwait1:			;Wartezeit: r16, 1 bis 255 msec
	push r16
	push r17
	clr r17
	rjmp miL1
milliwait255:
	ldi r16, 255
milliwait:			;Wartezeit: r17:r16 msec, min 1 max 1000
	push r16
	push r17
miL1:	push r18
	push r19
	mov r18, r10
loop1:	cp  r18, r10
	breq loop1		;1. Millisekunde abwarten (0 bis 1 ms)
	mov r18, r17		; Oberstes Bit von r17 nach r18 kopieren
	andi r17, 0x7F		; und in r17 loeschen.
	add r16, r10
	adc r17, r11		;sollzeit = Wartezeit + aktuelle Millisek
	subi r16, low(1001)
	sbci r17, high(1001)	;sollzeit -= 1; sollzeit -= 1000;
	brpl loop2		;wenn sollzeit>=0 -->
	subi r16, low(-1000)	;sonst haben wir 1000 zuviel subtrahiert
	sbci r17, high(-1000)	;also wieder 1000 addieren (soll -= -1000)
loop2:	mov r19, r10
	subi r19, -8
	cp r16, r10
	cpc r17, r11		;auf sollzeit warten
	cpc r16, r10		;falls zwischen 1. u 2. cp ein Interrupt war
	brne loop2
pop19ret: pop r19
pop18ret: pop r18
pop17ret: pop r17
pop16ret: pop r16
	  ret

admessung:	; AD-Wert lesen, ev. laufendes Mittel berechnen
		; Rueckgabewert: r16:r17
	push r0
	push r2
	push r3
	push r18
	push r19
	push r20
	lds r18, adrunflag
	tst r18			;wenn adwandler aus ist
	breq admalt		;--> nur alter Wert lesen
	lds r17, sensor1adwertL
	lds r16, sensor1adwertH
	lds r20, nmittel
	cpi r20, 2		; N<2 ?
	brcs mit1		; ja: --> ohne Mittelung
	lds r19, summeadwertL
	lds r18, summeadwertH
	mov r3, r16
	mov r2, r17		; r3:r2 = r16:r17
	rcall ringpop
	sub r19, r17
	sbc r18, r16		; summe -= N-ter vorheriger Wert
	mov r16, r3
	mov r17, r2
	rcall ringpush
	add r2, r19
	adc r3, r18		; summe += adwert
	sts summeadwertL, r2
	sts summeadwertH, r3
admalt:	lds r2, summeadwertL
	lds r3, summeadwertH
	lds r20, nmittel
	rcall divmodr20		; r3:r2 /= r20, r0=Rest
	mov r16, r3
	mov r17, r2		; Rueckgabewert=Summe/N
	rjmp mit2
mit1:	sts summeadwertL, r17
	sts summeadwertH, r16
mit2:
.ifdef LCDANZEIGE
	clr r18			;lcd-Position unveraendert lassen
	rcall lcd_r16r17
.endif
	pop r20
	pop r19
	pop r18
	pop r3
	pop r2
	pop r0
return:	ret
	
;-----------------------------------------------------------------------------
; Messpunkt vom vorherigen admessung speichern
; Eingabewert: r16:r17
; Rueckgabewert: -
messung:
	push r16
	push r17
	push r18
	push r19
	mov r18, r16
	mov r19, r17
.ifndef SEKSCHUETTLER
	lds r17, adrunflag	; nicht wirklich noetig ??
	tst r17			;wenn adwandler aus ist ??
	breq rpop19ret		;dann nichts machen.    ??
.endif
	ldi r17, 2		;LED 2 mal blinken
blinkloop1:
	ldi r16, 50
	rcall milliwait1	;50ms warten
	rcall ledportb
	rcall milliwait1	;50ms warten
	rcall ledportbclear
	dec r17
	brne blinkloop1		;Blinkloop braucht also 2*2*50 = 200ms
	mov r16, r18
	mov r17, r19
.if DATENFORMAT==0
	rcall bytespeichern2 	;H zuerst speichern
	mov r16, r17
	rcall bytespeichern2	;L danach speichern
.else ;if DATENFORMAT<=2
	rcall gepacktspeichern
.endif
	adiw r24, 1		;r25:r24 += 1, Anzahl gespeicherte Messpunkte
	mov r16, r18
	mov r17, r19
	ldi r18, 1		;Flag fuer lcd-Position erhoehen
.ifdef LCDANZEIGE
	rcall lcd_r16r17
.endif
rpop19ret:
	rjmp pop19ret

.ifdef FLASHVERSION
;-----------------------------------------------------------------------------
; Routinen zum Speichern im Flash
.ifndef ATMEGA328P
.define SELFPRGEN SPMEN
.define SPMCSR SPMCR
.endif

bytespeichern2:		;Byte speichern und den Zaehler r23:r22 erniedrigen
	subi r22, 1
	sbci r23, 0
			;Speicher voll? (sollte eigentlich nicht passieren)
	brcs return	;ja --> nichts machen
bytespeichern:
imflashspeichern:		;1 Byte im Flash speichern
	;; zuerst Zwischenspeichern und dann 2 Byte aufs mal speichern
	;; jeweils Page loeschen bevor das erste Word gespeichert wird.
	;; r16 = zu speicherndes Byte
	;; XH:XL = Zeiger im Flash, wird um 1 erhoeht
	.def zwischenspeicher=r8
; oder falls kein Register belegt werden soll:
;	.equ ZWISCHENSPEICHER=SRAMSTART+256 ;2Byte im SRAM
	push r0
	push r1
	push r16
	push r17
	push ZL
	push ZH
; zuerst noch in einen Puffer kopieren:
	lds ZL, zpufferL
	lds ZH, zpufferH
	clr r17
	cpi XL, 0
	cpc XH, r17		  ;erster Aufruf?
	breq zpu1
	ldi r17, high(flashpuffer+PAGESIZEB)
	cpi ZL, low(flashpuffer+PAGESIZEB)
	cpc ZH, r17		  ;oder schon bei Pufferende?
	brne zpu2
zpu1:	ldi ZL, low(flashpuffer)  ;ja: dann zpuffer auf 
	ldi ZH, high(flashpuffer) ;    Puffer-Anfang setzen
zpu2:	st Z+, r16
	sts zpufferL, ZL
	sts zpufferH, ZH
; fertig mit in einen Puffer kopieren.
	sbrc r9, 0		;schreiben wir in erste Page?
	rjmp imfL1		;nein-->
	ldi ZL, low(SRAMSTART)	;ja: auch noch ins SRAM kopieren
	ldi ZH, high(SRAMSTART)
	add ZL, XL
	adc ZH, XH
	st Z+, r16
	cpi ZH, high(SRAMSTART+PAGESIZEB) ;erste Page voll?
	brcs imfL1		    ;nein-->
	cpi ZL, low(SRAMSTART+PAGESIZEB) ;erste Page voll?
	brcs imfL1		    ;nein-->
	inc r9			    ;ja: r9=1 (bzw Bit0 in r9 auf 1 setzen)
imfL1:	mov ZL, XL
	mov ZH, XH
	sbrs ZL, 0		;Z ungerade?
	rjmp zwischenspeichern	;nein--> erstes Byte zwischenspeichern
wordspeichern:
	dec ZL			;immer auf gerade Adresse 2 Bytes speichern
	mov r0, zwischenspeicher ;oder lds r0, ZWISCHENSPEICHER
	mov r1, r16
	ldi r17, (1<<SELFPRGEN)
	rcall do_spm17
	adiw ZH:ZL, 2
	mov r17, ZL
	andi r17, PAGESIZEB-1	;wird beim naechsten Mal neue Page erreicht?
	brne incxandreturn	;nein-->
	subi ZL, low(PAGESIZEB)	;ja:  execute page write
	sbci ZH, high(PAGESIZEB) ;restore pointer
	ldi r17, (1<<PGWRT) + (1<<SELFPRGEN)
	rcall do_spm17
	rcall spm_ueberpruefen
	sbrs r16, 0
.ifdef LCDANZEIGE
	rcall print_error17
.endif
	rjmp incxandreturn
zwischenspeichern:
	mov zwischenspeicher, r16	;oder sts ZWISCHENSPEICHER, r16
	mov r17, ZL
	andi r17, PAGESIZEB-1	;neue Page?
	brne incxandreturn	;nein-->
	ldi r17, (1<<PGERS) + (1<<SELFPRGEN) ;ja: neue Page zuerst loeschen
	rcall do_spm17
incxandreturn:
	adiw XH:XL, 1
	pop ZH
	pop ZL
	pop r17
	pop r16
	pop r1
	pop r0
	ret
do_spm17:	;r17 ins SPMCR speichern und spm durchfuehren, r16 zerstoert
spmwait: in r16, SPMCSR	;; check for previous SPM complete
	sbrc r16, SELFPRGEN
	rjmp spmwait
	in r16, SREG
	push r16	;; disable interrupts if enabled, store status
	cli
;eewait:	sbic EECR, EEWE		; auf Ende von allfaelligem
;	rjmp eewait		; EEPROM Schreibzyklus warten
	out SPMCSR, r17	;; SPM timed sequence
	spm
	pop r16 ;; restore SREG (to enable interrupts if originally enabled)
	out SREG, r16
	ret
endeimflashspeichern:		;Rest der Page mit r16 auffuellen
	push r17
eflashloop:
	mov r17, XL
	andi r17, PAGESIZEB-1	;neue Page erreicht?
	breq eflashfertig	;ja-->
	rcall imflashspeichern
	rjmp eflashloop
eflashfertig:
	pop r17
	ret

spm_ueberpruefen:	;Flash ab Stelle ZH:ZL um PAGESIZEB Bytes nachpruefen
			;Rueckgabe: r16=okflag, r17=restliche Bytes, Z erhoeht
	push YL
	push YH
; Das ist zwar im Beispiel in AVR_Instruction_Set.pdf nicht drin,
; braucht es aber offenbar:
	ldi r17, (1<<RWWSRE)+(1<<SELFPRGEN)
	rcall do_spm17		; Flash-Speicher wieder zum Lesen freigeben

.ifdef DEBUGGING
	push ZL
	push ZH
	ldi YL, low(flashpuffer)
	ldi YH, high(flashpuffer)
	ldi r16, 0x21
	rcall lcd_go
	ld r16, Y+
	rcall lcd_r16hex
	ld r16, Y+
	rcall lcd_r16hex
	lpm r16, Z+
	rcall lcd_r16hex
	lpm r16, Z+
	rcall lcd_r16hex
	pop ZH
	pop ZL
.endif

	ldi YL, low(flashpuffer)
	ldi YH, high(flashpuffer)
	ldi r17, PAGESIZEB	; PAGESIZEB==64 beim Atmega8
rdloop:	lpm r0, Z+
	ld r16, Y+
	cpse r0, r16
	rjmp err
	dec r17
	brne rdloop
 	ldi r16, 1		; true == ok
	rjmp popyret
err:	clr r16			; false
popyret:pop YH
	pop YL
	ret
.endif

.ifdef LCDANZEIGE
string_L1:
	rcall lcd_write
print_string:	    ;String in ZH:ZL auf lcd ausgeben, Z erhoeht, r16 zerst.
	lpm r16, Z+
	tst r16
	brne string_L1
	ret

.ifdef STACKERROR
Stack:	.db "Stack",0
print_stack:
	push ZL
	push ZH
	ldi r16, 'S'
	sts errorflag, r16
	ldi ZL, low(Stack<<1)
	ldi ZH, high(Stack<<1)
	rcall print_string
	pop ZH
	pop ZL
	ret
.endif
;Error:	.db "Error",0
Error:	.db "Err",0
print_error17:			; r17 ausgeben und 'Error' auf LCD
	mov r16, r17
	rcall lcd_r16dez
print_error:
	ldi r16, 'E'
	sts errorflag, r16
	push ZL
	push ZH
	ldi ZL, low(Error<<1)
	ldi ZH, high(Error<<1)
	rcall print_string
	pop ZH
	pop ZL
	ret
.endif

.ifdef LCDANZEIGE
;-----------------------------------------------------------------------------
; Anzeige auf LCD-Modul
microwait:			; etwa r16 usec warten
	push r16
	rjmp mic_L1
microwait1:			; etwa 1 usec warten
	push r16
	push r17
	ldi r16, 4
	rjmp mic_L2
;microwait10:			; test
;	push r16
;	ldi r16, 10
mic_L1: push r17
	add r16, r16
	add r16, r16
mic_L2:	dec r16
	brne mic_L2
	pop r17
	pop r16
	ret

.ifdef TEXT_LCD

lcd_ziffer:		; void lcd_ziffer(int n) ;fuer 0<=n<=9
	ori r16, 0x30
	cpi r16, 0x3A	; oder fuer Hex-Ziffern bis n<=15
	brcs lcd_write
	subi r16, -7	; fuer Hex-Ziffern z.B. 3A --> 41=='A'
lcd_write:		;void lcd_write(char r16=c)
	sbi PORTD, 2
	rjmp lcd_send
lcd_cmd:	;void lcd_cmd(char cmd)
	cbi PORTD, 2
	rcall microwait1
lcd_send:	; lcd_send(char data) ;r16=data
	push r17
	push r18
	in r17, PORTD	;rs=PORTD
	andi r17, 4	;rs &= 4;
	ori r17, 3		; fuer Pullup-Widerstaende
	mov r18, r16	;char tmp=data
	andi r18, 0xF0	;tmp &= 0xF0
	or r18, r17	;tmp |= rs
	out PORTD, r18	;PORTD=tmp
	rcall microwait1
	sbi PORTD, 3	;PORTD |= (1<<3); //sbi(PORTD,3)
	rcall microwait1
	cbi PORTD, 3	;PORTD &= ~(1<<3); //cbi(PORTD,3)
	mov r18, r16	;tmp=data
	andi r18, 0x0F	;tmp &= 0x0F
	swap r18	;tmp <<= 4
	or r18, r17	;tmp |= rs
	out PORTD, r18	;PORTD=tmp
	rcall microwait1
	sbi PORTD, 3	;PORTD |= (1<<3); //sbi(PORTD,3)
	rcall microwait1
	cbi PORTD, 3	;PORTD &= ~(1<<3); //cbi(PORTD,3)
	ldi r16, 1
	rcall milliwait1
	pop r18
	pop r17
	ret

lcd_clear:
	push r16
	ldi r16, 0x01
lcd_cmd_wait2:
	rcall lcd_cmd
	ldi r16, 2
	rcall milliwait1
	pop r16
	ret
lcd_home:
	push r16
	ldi r16, 0x02
	rjmp lcd_cmd_wait2

lcd_on:	push r16
	ldi r16, 0x0E
lcd_cmd_2:
	rcall lcd_cmd
	pop r16
	ret
lcd_off:
	push r16
	ldi r16, 0x08
	rjmp lcd_cmd_2

lcd_init:
	push r16
	ldi r16, 0xFC
	out DDRD, r16	;Port D obere 6 Bits auf Ausgang
	ldi r16, 3	;PORTD=0; (3 = +PullupWiderstaende)
	out PORTD, r16
	ldi r16, 50
	rcall milliwait	;//Warten bis LCD-Controller gebootet hat
	ldi r16, 0x23		; (3 = +PullupWiderstaende)
	out PORTD, r16	;PORTD=0x20; //4-Bit-Modus einschalten
	rcall microwait1
	sbi PORTD, 3	;PORTD |= (1<<3); //sbi(PORTD,3); //Enable-Impuls
	rcall microwait1
	cbi PORTD, 3	;PORTD &= ~(1<<3); //cbi(PORTD,3);
	rcall milliwait5
	ldi r16, 0x28	;lcd_cmd(0x28);
	rcall lcd_cmd	;  //Funktions-Set: 2 Zeilen, 5*7 Matrix, 4Bit
	rcall lcd_off
	rcall lcd_clear
	ldi r16, 0x06	;lcd_cmd(0x06); // Entry Mode
	rcall lcd_cmd
	rcall lcd_on
	ldi r16, 0x11
	sts lcdpos, r16
	pop r16
	ret

lcd_go:
	push r17
	mov r17, r16
	swap r16
	rcall lcd_goto
	pop r17
	ret
lcd_goto:		;lcd_goto(r16=row,r17=col)
	dec r16		;row--
	andi r16, 0x01	;row &= 0x01; //nur Zeile 0 oder Zeile 1 zulassen
	swap r16
 	add r16, r16		
 	add r16, r16	;row <<= 6; //Zeile nach Bit 6 bringen
	dec r17		;col--;
	andi r17, 0x0F	;col &= 0x0F; //nur Bereich 0 bis 15 zulassen
	or r16, r17	;char tmp=row|col; //Adresse bilden
	ori r16, 0x80	;tmp |= 0x80; //Cursor setzen
	rcall lcd_cmd	;lcd_cmd(tmp);
	ret
.endif

lcd_r1:		;Ziffer in r1, und wenn r2 gesetzt 'M', auf LCD anzeigen
	push r16
	push r17
	ldi r16, 0x11
	rcall lcd_go
	mov r16, r1
	andi r16, 0x0F
	rcall lcd_ziffer
	ldi r16, ' '
	rcall lcd_write
	ldi r16, ' '
	tst r2			; Mittelwertflag gesetzt?
	breq _ohne		; nein-->
	ldi r16, 'M'
_ohne:	rcall lcd_write
	rjmp pop17ret

lcd_r16dez:
	ldi r17, 10
	rjmp lcd_r16
lcd_r16hex:
	ldi r17, 16
lcd_r16:	;r16 auf LCD, Dezimal oder Hexadezimal ohne fuehrende Nullen
		;Eingabe: r16=Zahl, r17=10 oder r17=16 fuer Hexadezimal
	push r0
	push r2
	push r3
	push r16
	push r17
	push r20
	push XL
	push XH
	clr r3
	mov r2, r16
	mov r20, r17		;Basis (10 oder 16)
	ldi XL, low(lcdn0)
	ldi XH, high(lcdn0)
	ldi r17, 1		;Zaehler fuer Anzahl Ziffern
	rjmp lc16L2
lc16L1:	rcall divmodr20		;durch 10 dividieren (oder durch 16)
	st X+, r0
	inc r17
lc16L2:	cp r2, r20		;Zahl>=10 ?
	brcc lc16L1		;ja--> weiter dividieren
	mov r16, r2
	rcall lcd_ziffer
	rjmp lc16L4
lc16L3:	ld r16, -X
	rcall lcd_ziffer
lc16L4:	dec r17
	brne lc16L3
	pop XH
	pop XL
	rjmp popr20ret

lcd_r16r17:	;Zahl r16:r17 auf LCD anzeigen, Wenn r18==1 lcdpos erhoehen
	push r0
	push r2
	push r3
	push r16
	push r17
	push r20
	mov r3, r16
	mov r2, r17
.ifdef GRAFIK_LCD
	rcall punktgrafischdarstellen
.endif
	lds r16, lcdpos
	rcall lcd_go
lcd_L0:	ldi r20, 10
	rcall divmodr20
	sts lcdn0, r0
	rcall divmodr20
	sts lcdn1, r0
	rcall divmodr20
	sts lcdn2, r0
	rcall divmodr20
	sts lcdn3, r0
	rcall divmodr20
	mov r16, r0
	tst r16			; wenn fuehrende Ziffer keine 0 ist..
	breq lcd_L1
	ori r16, 0x30		; ..dann die Ziffer selbst..
	rjmp lcd_L2
lcd_L1:	ldi r16, 0x20		; ..sonst ' ' ausgeben.
lcd_L2:	rcall lcd_write
	lds r16, lcdn3
	rcall lcd_ziffer
	lds r16, lcdn2
	rcall lcd_ziffer
	lds r16, lcdn1
	rcall lcd_ziffer
	lds r16, lcdn0
	rcall lcd_ziffer
.ifdef GRAFIK_LCD
	ldi r16, ' '		; falls kein Cursor, nur Leerzeichen
	tst r18			; r18==0 ?
	brne lcd_g1
	ldi r16, '_'		; ja: Cursor darstellen
lcd_g1:	rcall lcd_write
	ldi r16, ' '
	rcall lcd_write
.endif
	tst r18			; r18==0 ?
	breq lcd_L4		; ja--> lcdpos nicht veraendern
	lds r16, lcdpos
	cpi r16, 0x11		; sind wir auf 1.Zeile 1.Zeichen ?
	breq lcd_L11		; ja-->
.ifdef GRAFIK_LCD
;; Darstellung auf Grafik-LCD in nur einer Textzeile
;; Zeile hat 21 Zeichen, Darstellungsbeispiel:
;; "123456789 123456789 1"
;; "1023   1023   1023_  "
	cpi r16, 0x18
	breq lcd_L18
	ldi r16, 0x11		; 1.Zeile 1.Zeichen
	rjmp lcd_L3
lcd_L11:ldi r16, 0x18		; 1.Zeile 8.Zeichen
	rjmp lcd_L3
lcd_L18:ldi r16, 0x1F		; 1.Zeile 15.Zeichen
;	rjmp lcd_L3
.else
;; Darstellung auf Text-LCD in 2 Zeilen
	cpi r16, 0x18
	breq lcd_L18
	cpi r16, 0x21
	breq lcd_L21
	ldi r16, 0x11
	rjmp lcd_L3
lcd_L11:ldi r16, 0x18
	rjmp lcd_L3
lcd_L18:
;	ldi r16, 0x11		;Test mit nur 1 Zeile
	ldi r16, 0x21
	rjmp lcd_L3
lcd_L21:ldi r16, 0x28
.endif
lcd_L3:	sts lcdpos, r16
lcd_L4:
popr20ret:
	pop r20
	pop r17
	pop r16
popr3ret:	
	pop r3
	pop r2
	pop r0
	ret

.ifdef GRAFIK_LCD

lcd_ziffer:		; void lcd_ziffer(int n) ;fuer 0<=n<=9
	ori r16, 0x30
	cpi r16, 0x3A	; oder fuer Hex-Ziffern bis n<=15
	brcs lcd_z1
	subi r16, -7	; fuer Hex-Ziffern z.B. 3A --> 41=='A'
lcd_z1:	rjmp glcd_writec

.include "glcd.asm"

lcd_init:
	push r16
	clr r16
	sts grafikposx, r16
	ldi r16, 64
	sts grafikposy, r16
	rcall glcd_init2
	ldi r16, 0x11
	sts lcdpos, r16
.ifdef VERSIONSANZEIGE
	ldi r16, 0x17
	rcall lcd_go
	rcall print_version
.endif
	pop r16
	ret

lcd_write:rjmp glcd_writec

lcd_go:	push r17
	mov r17, r16
	andi r17, 0x0F
;	swap r16
;	andi r16, 0x0F
;	subi r16, -6	;7. und 8. Zeile statt 1. und 2. benutzen
	ldi r16, 8	;immer 8.Zeile benutzen (mehr Platz fuer Grafik)
	rcall lcd_goto
	pop r17
	ret

lcd_goto:
	dec r16
	dec r17
	rjmp glcd_goto

punktgrafischdarstellen:	;r3:r2=Messwert, r18==1 -> position erhoehen
	push r18		;Rueckgabe: r16, r17, r20 zerstoert
	push r19
	mov r20, r18
	lds r17, grafikposy
	cpi r17, 64		;in gueltigem Wertebereich?
	brcc pun1		;nein-->
	lds r16, grafikposx
	clr r18
	rcall glcd_punkt	;ja: alter Punkt loeschen
pun1:	mov r16, r3
	mov r17, r2		;y von Wertebereich 0..1023 auf 8..63 umrechnen
	lds r18, skalfaktor
pun1a:	lsr r18			;durch Skalierfaktor dividieren
	breq pun1b		;wenn (r18/=2)==0 --> fertig
	cpi r18, 1		;letztes Mal durch 2 dividieren?
	brne pun1a1		;nein-->
	subi r17, -1		;ja: 1 addieren fuer bessere Rundung
	sbci r16, -1
pun1a1:	lsr r16
	ror r17			;y/=2 (Skalierfaktor nur Werte 2^n erlaubt)
	rjmp pun1a
pun1b:	ldi r19, 0
	ldi r18, 56		;Benutzbarer Grafikbereich 64-8=56
pun1c:	cpi r16, 0
	cpc r17, r18		;y < 56?
	brcs pun1d		;ja-->
	subi r16, 0
	sbc r17, r18		;nein: 56 subtrahieren
	inc r19			;Anzahl Bereichueberschreitungen zaehlen
	rjmp pun1c
pun1d:	cpi r19, 2		;mindestens 2 mal Bereich ueberschritten?
	brcc ska_plus		;ja--> Skalierfaktor erhoehen
	tst r19			;Bereich nie ueberschritten und ..
	brne pun2
	cpi r17, 19		;.. Wert im untern Drittel des Grafikbereichs?
	brcc pun2		;nein--> weiter
ska_minus:			;ja: Skalierfaktor verkleinern
	lds r18, skalfaktor
	lsr r18
	brcs pun2		;wenn Skalierfaktor==1 --> nicht verkleinern
	rjmp ska_neu		;sonst kleinerer Faktor speichern
ska_plus:			;Skalierfaktor erhoehen
	lds r18, skalfaktor
	lsl r18
	cpi r18, 16+1		;neuer Faktor > maximaler Faktor?
	brcc pun2		;ja--> nicht speichern
ska_neu:sts skalfaktor, r18	;neuer Faktor speichern
pun2:	subi r17, -8		;8 addieren um Textzeile nicht zu uebermalen
	sts grafikposy, r17
	lds r16, grafikposx
	ldi r18, 1
	rcall glcd_punkt	;aktueller Punkt zeichnen
.ifdef MARKER
	clr r18
	rcall marker_zeichnen	;alter Marker loeschen
	push r17
	push r16
	inc r16			; neuer Marker x eine Position weiter setzen
	andi r16, 127		; x auf maximal 127 beschraenkt
	subi r17, 2		; y 2 Punkte weiter unten zum dann 5 Punkte zeichnen
	cpi r17, 8		; im Textbereich?
	brcc mark_1		; nein-->
	ldi r17, 8		; ja: Marker oberhalb des Textbereichs
	rjmp mark_2
mark_1:	cpi r17, 64-5		; noch genug Platz zum 5 Punkte zeichnen?
	brmi mark_2		; ja-->
	ldi r17, 64-5
mark_2:	sts markposy, r17
	sts markposx, r16
	ldi r18, 1
	rcall marker_zeichnen
	pop r16
	pop r17
.endif
	cpi r20, 1
	brne punktg_ret
	lds r16, grafikposx
	inc r16			;x-Position erhoehen
	andi r16, 127		;auf 128 beschraenkt
	sts grafikposx, r16
	ldi r17, 64
	sts grafikposy, r17	;auf ungueltigen Wert setzen
	ldi r17, 8		;erster Grafikpunkt oberhalb Textzeile
	clr r18
pun3:	rcall glcd_punkt	;Grafikpunkte bei neuem x-Wert loeschen
	inc r17
	cpi r17, 64
	brne pun3
.ifdef MARKER
	sts markposy, r18	;Markerposition als ungueltig markieren
.endif
punktg_ret:
	pop r19
	pop r18
	ret

.ifdef MARKER
marker_zeichnen:		;r18 = Farbe (0=loeschen, 1=zeichnen)
	push r16
	push r17
	push r19
	lds r17, markposy
	tst r17			;Markerposition gueltig?
	breq mark_ret		;nein--> nichts tun
	lds r16, markposx
	ldi r19, 5		;5 Punkte grosser Marker
mark_loop:
	rcall glcd_punkt	;Marker zeichnen/loeschen
	inc r17
	dec r19
	brne mark_loop
	tst r18			;Marker geloescht?
	brne mark_ret
	sts markposy, r18	;ja: Markerposition als ungueltig markieren
mark_ret:
	pop r19
	pop r17
	pop r16
	ret
.endif
	
.endif

.endif
;------- Ende LCD-Anzeige ------------

;-----------------------------------------------------------------------------
; divmodr20
;  16Bit vorzeichenlose Zahl dividieren durch 8Bit vorzeichenlosen Teiler
; Eingabe: r3:r2 = Zahl (16Bit), r20 = Teiler (8Bit) (unveraendert)
; Ausgabe: r3:r2 = Resultat, r0 = Rest
;-----------------------------------------------------------------------------
divmodr20:
	push r17
	clr r0
	ldi r17, 16
divL1:	lsl r2
	rol r3
	rol r0
	cp r0, r20
	brcs divL2
	sub r0, r20
	inc r2
divL2:	dec r17
	brne divL1
	pop r17
	ret

.ifdef MULDIV
mul16:		;r3:r2:r1:r0 = r1:r0 * r3:r2
	push r16
	push r17
	push r18
	push r19
	push r6
	push r7
	mov r16, r0
	mov r17, r1		; AB
	mov r18, r2
	mov r19, r3		; CD
	mul r17, r19		;A*C
	mov r2, r0		; EFGH = AB*CD --> EF=A*C GH=B*D EFG+=A*D+B*C
	mov r3, r1		; r3r2 = EF=A*C
	mul r16, r18		;B*D
	mov r6, r0
	mov r7, r1		; r6r7 = GH=B*D
	mul r17, r18		;A*D
	clr r17
	add r5, r0		; EFG+=A*D
	adc r2, r1
	adc r3, r17
	mul r16, r19		;B*C
	add r5, r0		; EFG+=B*C
	adc r2, r1
	adc r3, r17
	mov r0, r6
	mov r1, r7
	pop r7
	pop r6
	rjmp pop19ret

div16:			;r3:r2:r1:r0 /= r17:r16, r17:r16 = Rest
	push r18
	push r19
	push r20
	clr r18
	clr r19
	ldi r20, 32
div16L1:lsl r0
	rol r1
	rol r2
	rol r3
	rol r18
	rol r19
	cp  r18, r16
	cpc r19, r17
	brcs div16L2
	sub r18, r16
	sbc r19, r17
	inc r0
div16L2:dec r20
	brne div16L1
	mov r16, r18
	mov r17, r19		;r17:r16 = Rest
	pop r20
	pop r19
	pop r18
	ret
.endif

.ifdef MAGNETRUEHRER
magnetruehrer:			; Magnetruehrer ein/aus-schalten
;r21:r20 = Sollzeit  r13:r12 = aktuelle Min:Sek
	push r16
	push r20
	push r21
	lds r16, magnetschaltzeit
	tst r16
	breq _magret
	sub r20, r16		;Sollzeit fuer "Magnetruehrer ausschalten"
	sbci r21, 0
	brcc _magL1
	ldi r21, 59
_magL1:	cp r20, r12		;Sollzeit Sekunden erreicht?
	cpc r21, r13		;Sollzeit Minuten erreicht?
	brne _magret
	cbi PORTB, 5		;Magnetruehrer aus
_magret:pop r21
	pop r20
	pop r16
	ret
.endif

.ifdef SCHUETTLER
schuetteln:			; Schuettler ev laufen lassen
;r21:r20 = Sollzeit  r13:r12 = aktuelle Min:Sek
	push r16
	push r20
	push r21
.ifdef SEKSCHUETTLER
	lds r16, schuettmodus
	tst r16			;Schuettelmodus eingestellt?
	breq _magL0		;nein--> normal weiter
	lds r20, schuettsoll	;Sollzeit Sekunden setzen
	mov r21, r13		;aktuelle Minute setzen
	rjmp _magL1
_magL0:	
.endif
	subi r20, 3		;Sollzeit -= 3 Sek Schuettelzeit
	brcc _magL1
	subi r20, -60		; 60 Sek zuzaehlen
	subi r21, 1		; dafuer 1 Min abziehen
	brcc _magL1
	ldi r21, 59
_magL1:	sub r20, r12		;Sollzeit Sekunden erreicht?
	sbc r21, r13		;Sollzeit Minuten erreicht?
	brne _magret		;nein-->
	sts adrunflag, r21	;adwandler_aus
	sbi PORTB, 4		;Puls fuer Schuettelvorgang senden
	rcall milliwait5
	cbi PORTB, 4
_magret:
	pop r21
	pop r20
	pop r16
	ret
.endif

.ifdef VERSIONSANZEIGE
ScattVers:
	.db "Scatter "
Vers:	.db "Version  1.08  ",0
print_version:
	push ZL
	push ZH
	ldi ZL, low(Vers<<1)
	ldi ZH, high(Vers<<1)
	rcall print_string
	pop ZH
	pop ZL
	ret
.endif
