/* rflash.cc  Flash von Microkontroller auslesen   letzte Aenderung 13.12.2010 */
#define VERSION "Version 0.03"
/*
History:
12.11.2010	Erstellung (RP)
17.11.          Wahl zwischen Flash und EEPROM, ins EEPROM auch schreiben
24.11.          Nach Einlesen vom Flash: Auswertung der Daten und als .xy speichern
13.12.10  0.03  Anpassung an Cygwin (Windows)

*/

//nur eine der 3 folgenden Zeilen nicht auskommentiert:
#define LINUX  1
//#define MACOSX 1
//#define CYGWIN 1 //Version fuer Windows

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <termios.h>
#include <unistd.h>

/************************* Vordeklarationen ***************************/
void entpacken(FILE *fp,FILE *fp2);
double umrechnen(int sensor,int variante,int wert);
//int rueckumrechnen(int sensor,int variante,double y);
void eeconvert(const char *quellname,const char *zielname);

/*************************** kleinkram ***************************/
inline int idfix(double x) {return int((x>=0.) ? x+0.5 : x-0.5);} //Runden

int getline(FILE *fp,char *s,int lim)
{		/* liest eine Textzeile oder maximal lim Zeichen */
		/* und ersetzt den Zeilentrenner durch 0         */
 int c;
 while(--lim && (c=getc(fp))!=EOF && c!='\n')
	*s++ = c;
 *s='\0';
 return (c!=EOF);	/* TRUE wenn erfolgreich, FALSE wenn Fileende */
}

/**************** Routinen zur Parameterauswertung ********************/
#define MAXARG 2
static char argflag[128];
void setargflags(char *s)
{
 int c;
 while(c= *s++)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
  }
}

/*********************** Serielle Schnittstelle ***********************/
#include <sys/errno.h>
#include <sys/ioctl.h>

// Voreinstellung des Pfades zum Seriellport:
#ifdef MACOSX
static char sername[80]="/dev/tty.serial-0002"; //auf MacOSX
//static char sername[80]="/dev/tty.SLAB_USBtoUART"; //MacOSX mit neuerem Silab-Treiber
#endif
#ifdef LINUX
//static char sername[80]="/dev/ttyS0"; //richtiger Seriellport mit Linux
static char sername[80]="/dev/ttyUSB0"; //mit Linux-Kubuntu
#endif
#ifdef CYGWIN
static char sername[80]="/dev/com1"; //mit Cygwin unter Windows
#endif

static int baudrate=38400; //Voreinstellung der Baudrate oder 0 fuer schon eingestellte Baudrate

static int fdr,fdw;
static int baudcodliste[]={50,B50, 75,B75, 110,B110, 134,B134, 150,B150,
   300,B300, 600,B600, 1200,B1200, 1800,B1800, 2400,B2400, 4800,B4800,
   9600,B9600, 19200,B19200, 38400,B38400,
   57600,B57600, 115200,B115200,
   230400,B230400,
   /* 460800,B460800, 500000,B500000, 576000,B576000,
   921600,B921600, 1000000,B1000000, 1152000,B1152000, 1500000,B1500000,
   2000000,B2000000, 2500000,B2500000, 3000000,B3000000, 3500000,B3500000,
   4000000,B4000000, */
   0,0};

int ibaud(int x) //Baudcode in Baudrate umrechnen
{
 int *co,a;
 for(co=baudcodliste;(a= *co++)!=0;co++)
    if(x== *co) return a;
 printf("Fehler: unbekannter Baudratencode %d\n",x);
 return 0;
}

int baudi(int x) //Baudrate in Baudcode umrechnen
{
 int *co,b,min=0;
 for(co=baudcodliste;(b= *co++)>0;co++)
     if(x==b) return *co;
     else if(x>b) {min= *co;}
 printf("Warnung: unbekannte Baudrate %d, auf %d abgerundet\n",x,min);
 return min;
}

static struct termios origbuf;

void noecho(int fd)
{
 int i,a,b;
 struct termios buf;
 if((a=tcgetattr(fd,&buf))<0)
   {printf("tcgetattr() Error %d\n",a); return;}

 memcpy(&buf,&origbuf,sizeof(struct termios));
 buf.c_cflag = CS8; //8 Bit ohne Parity
 buf.c_cflag |= CREAD;  //Enable input
 buf.c_cflag |= CLOCAL; //Ignore modem cntl
 buf.c_iflag = IGNBRK;
 buf.c_lflag = 0; //no echo input
 for(i=0; i<NCCS; i++)   // REF:JS12052003 Change to clear all option chars
   buf.c_cc[i] = 0;
 buf.c_cc[VMIN] = 0;     // Non blocking input
 buf.c_cc[VTIME] = 0;    // REF:JS12052003 Set to not wait

 if(baudrate!=0)
     {cfsetispeed(&buf,baudi(baudrate));
      cfsetospeed(&buf,baudi(baudrate));
     }

 if((a=tcsetattr(fd,TCSANOW,&buf))<0)
   {printf("tcsetattr() Error %d\n",a); return;}

 //a=cfgetispeed(&buf); b=cfgetospeed(&buf);
 //printf("Baudraten: %d %d\n",ibaud(a),ibaud(b)); //test
}

int seriell_open()
{
 fdr=open(sername, O_NONBLOCK | O_RDWR | O_NOCTTY);
 if(fdr<0) return fdr;
 //if(fdr<0) {printf("Fehler: fdr = %d\n",fdr); exit(0);}
	   //wenn dieser Fehler kommt sollte /dev/ttyS0 ueberprueft werden
	   //Zugriffsberechtigung freigeben: chmod a+rw /dev/ttyS0
	   //oder vielleicht /dev/ttyS1 oder /dev/tty.*

#ifdef CYGWIN
 //in Cygwin fehlt TIOCEXCL. Unter Linux ist das in einer 
 //include-Datei als "#define TIOCEXCL 0x540C" gesetzt.
#else
 ioctl(fdr, TIOCEXCL); //set for exclusive OPEN
#endif
 noecho(fdr);
 fdw=fdr;
 return 0;
}

void seriell_close()
{
 if(fdr != -1) tcsetattr(fdr, TCSANOW, &origbuf);
 close(fdr);
}

const char* ser_error()
{
 static char buf[16];
 switch(errno)
  {case EBADF: return "EBADF";
   case EFAULT: return "EFAULT";
   case EIO: return "EIO";
   case EINTR: return "EINTR";
   case EINVAL: return "EINVAL";
   case EAGAIN: return "EAGAIN";
   case ENXIO: return "ENXIO";
   case ENOENT: return "ENOENT";
  }
 sprintf(buf,"%d",errno);
 return buf;
}

#ifdef CYGWIN
void ser_read(char *pc)
{
 int i;
 static char buffer[64],*p=NULL;
 static int j=0;
 if(j==0)
   {while((i=read(fdr,buffer,64))==0 || (i<0 && errno==EAGAIN)) ;
    if(i<0) {printf("ser_read() Error: %s\n",ser_error()); return;}
    j=i;
    p=buffer;
   }
 *pc = *p++;
 --j;
}
#else
void ser_read(char *pc)
{
 int i;
 while((i=read(fdr,pc,1))==0 || (i<0 && errno==EAGAIN)) ;
 if(i<0) printf("ser_read() Error: %s\n",ser_error());
}
#endif

int seriell_readc()
{
 char c;
 ser_read(&c);
 return c;
}

void seriell_readline(char *str,int max)
{
 int i;
 char *s,c;
 for(i=1,s=str;i<max;)
   {
    ser_read(&c);
    if(c=='\n') break;
    //if(c==0) printf("\\0");//test
    else
      {*s++ = c; i++;}
   }
 if(s!=str && s[-1] == '\r')
   {--s;
    //printf("CR empfangen\n");//test
   }
 *s=0;
}

void ser_write(char* wbuf,int n)
{
 int i;
 while(n>0)
  {while((i=write(fdw,wbuf,n))==0 || (i<0 && errno==EAGAIN)) ;
   if(i<0) printf("ser_write() Error: %s\n",ser_error());
   if(i<n) wbuf = &wbuf[i];
   n -= i;
  }
}

void seriell_writeline(const char *str)
{
 int n=strlen(str);
 char *wbuf=new char[n+3];
 sprintf(wbuf,"%s\n",str);
 ser_write(wbuf,n+1);
 delete wbuf;
}

int ser_verbindungstest()
{
 char zeile[40]; zeile[0]=0;
 seriell_writeline("v"); //Version vom Scatter-Geraet anfordern
 seriell_readline(zeile,40);
 //if(argflag['V'])
  printf("%s\n",zeile);
 if(*zeile!=':')
   {seriell_close();
    return -1; //Fehler
   }
 return 0; //ok
}

/************************* Hauptprogramm ******************************/
int hexzahl(const char *s)
{
 int c1,c2;
 c1 = *s++; if(c1>='A') c1 += 10-'A'; else c1 -= '0';
 c2 = *s;   if(c2>='A') c2 += 10-'A'; else c2 -= '0';
 return (c1<<4)+c2;
}

main(int argc,char *argv[])
{
 char name[80],zeile[80], *defaultflags="rf";
 int i,j,wahl=1,c=0;
 FILE *fp,*fp2;
 name[0]=0;
 for(j=0,i=1;i<argc;i++)
	{if((c= *argv[i])=='-' || c=='?') setargflags(argv[i]);
	 else	{if(++j==1) strcpy(name,argv[i]);
	         else if(j==2) sscanf(argv[i],"%d",&baudrate);
	}	}
 if(argflag['?'] || j>MAXARG)
	{printf("rflash  %s\n",VERSION);
	 printf("Usage: rflash [-flags] [/dev/ttyUSB1] [Baudrate]\n");
	 printf("   flags: re = read EEPROM\n");
	 printf("          rf = read FLASH\n");
	 //printf("          r = response after each line\n");
	 //printf("          we = write EEPROM (not yet supported)\n");
	 printf("           v = verbose\n");
	 printf("           m = use minutes for x axis\n");
	 printf("           s = use hours for x axis\n");
	 printf(" default: rflash -%s %s %d\n",defaultflags,sername,baudrate);
	 exit(0);
	}
 if(!(argflag['W'] || argflag['E'] || argflag['F'])) setargflags(defaultflags);
 if(j>=1)
     {if(isdigit(*name)) sprintf(sername,"/dev/ttyUSB%c",*name);
      else sprintf(sername,"%s",name);
     }
 //printf("opening '%s'\n",sername); //test
 while(seriell_open()!=0 || ser_verbindungstest()!=0)
  {
#ifdef CYGWIN
   static int j=1;
   if(++j<=4) sprintf(sername,"/dev/com%d",j);
   else {
#endif
   printf("cant connect to \"%s\"\ntry with another device: ",sername);
   scanf("%s",sername);
#ifdef CYGWIN
        }
#endif
  }

 /* Einfaches Uebertragungsprotokoll:
      Kommandos: getf = Flash lesen, gete = EEPROM lesen
      Microkontroller soll dann gesamtes Flash oder gesamtes EEPROM als Hex-Datei senden.
      Kommando pute = EEPROM schreiben, danach Daten als Hex-Datei senden. Antwort vom
      Microkontroller ist dann Anzahl Fehler (also 0 wenn alles ok).
  */
 if(argflag['W'])
   {int fehler=0;
    printf("hexfile to send:"); getline(stdin,zeile,80);
    fp=fopen(zeile,"r");
    if(fp==NULL) {printf("'%s' not found.\n",zeile); exit(0);}
    seriell_writeline("pute");
    while(getline(fp,zeile,80))
      {seriell_writeline(zeile);
       //printf("gesendet: '%s'\n",zeile);//test
       if(seriell_readc()!='0') fehler++;
       seriell_readc();
      }
    zeile[0]=0;
    seriell_readline(zeile,80);
    sscanf(zeile,"%d",&fehler);
    if(fehler==0) printf("ok.\n");
    else printf("%d errors.\n",fehler);
   }
 else
   {int fehler=0,i,n,pruefsumme;
    const char *rawfile="scatter.raw";
    fp2=fopen(rawfile,"wb");
    if(fp2==NULL) {printf("cant open '%s'\n",rawfile); seriell_close(); exit(0);}
    if(argflag['E']) seriell_writeline("gete");
    else             seriell_writeline("getf");
    do {fehler=0;
        seriell_readline(zeile,80);
	if(argflag['V']) printf("%s\n",zeile);
	if(zeile[0]!=':') fehler=1;
	n = hexzahl(&zeile[1]);
	for(i=0,pruefsumme=0;i<n+5;i++) //Uber n Datenbytes + 5 weitere die Pruefsumme testen
	  pruefsumme += hexzahl(&zeile[1+2*i]);
	if((pruefsumme&0xFF)!=0) fehler=1;
	else
	  {for(i=0;i<n;i++)
	      putc(hexzahl(&zeile[9+2*i]),fp2);
	  }
	if(fehler)
	  {printf("wrong data: '%s' - ",zeile);
	   if(argflag['R']) printf("reading again\n");
	   else printf("ignored\n");
	  }
	if(argflag['R']) seriell_writeline(fehler==0?"0":"1");
       }
    while(strncmp(&zeile[7],"00",2)==0);
    fclose(fp2);
    eeconvert(rawfile,"scatter.xy");
   }
 seriell_close();
 return 0;
}/* ende von main */

void eeconvert(const char *quellname,const char *zielname)
{
 FILE *fp,*fp2;
 if(fp=fopen(quellname,"r"))
  {if(fp2=fopen(zielname,"w"))
    {
     entpacken(fp,fp2);
     fclose(fp2);
    }
   else printf("cant crate '%s'\n",zielname);
   fclose(fp);
  }
 else printf("cant find '%s'\n",quellname);
}

/*********************** Kopie aus eeconvert.cc *********************************/
void entpacken(FILE *fp,FILE *fp2)
{
/*
; 1 Byte DATENFORMAT
; 1 Byte SENSORTYP
; 1 Byte ANSCHLUSSVARIANTE
; 1 Byte Messperiode = Anzahl Sekunden zwischen 2 Messungen
; 2 Byte N = Anzahl Messpunkte
;Wenn DATENFORMAT==0: N*2 Byte ungepackte Daten
;sonst je nach Packvariante weniger als N*2 Bytes
;
;DATENFORMAT==1:
;       1X -> 4Bit, davon 3Bit Differenzwert (-4 ... +3)
;       01 -> 8Bit, davon 6Bit Differenzwert (-36 ... -5 und +4 ... +35)
;       00 -> 12Bit, davon 10Bit Absolutwert vorzeichenlos (0 ... 1023)
;DATENFORMAT==2:
; Wie 1 aber mit 14Bit-Absolutwerten:
;       00 -> 16Bit, davon 14Bit Absolutwert vorzeichenlos (0 ... 16383)
; Die Werte sind GradCelsius*10 (z.B. 357 bedeutet 35.7 Grad)
*/
 unsigned char format,sensor,variante,periode;
 int npunkte;//Anzahl Messpunkte
 unsigned char c1,c2,b1,b2,b3,b4,bcount=0;
 int i,wert=0,diff;
 double x=0,y=0,x2;
 format=getc(fp); sensor=getc(fp); variante=getc(fp); periode=getc(fp);
 if(argflag['V'])
  printf("Data format=%d, Sensor=%d, Connection variant=%d, Period=%d\n",
	 format,sensor,variante,periode);
 c1=getc(fp); c2=getc(fp); npunkte=(c1<<8)+c2;
 if(argflag['V']) printf("npoints=%d\n",npunkte);
 for(x=0,i=0;i<npunkte;i++,x+=periode)
 {if(format==0)
  {//ungepackt
   c1=getc(fp); c2=getc(fp);
   wert=(c1<<8)+(c2&0xFF);
  }
  else //if(format<=2)
  {if(bcount==0)
     {c1=getc(fp);
      b1=(c1>>4)&0x0F; //erstes Nibble
      b2=c1&0x0F; //zweites Nibble
      bcount=2;
     }
   if(b1&0x8)
    {diff=b1&0x7; if(diff&0x4) diff-=8;
     if(--bcount==1) b1=b2;
     wert+=diff;
    }
   else if((b1&0xC)==0x4)
    {if(bcount==1)
      {c1=getc(fp); b2=(c1>>4)&0x0F;}//2Nibble nachladen
     diff=((b1&0x3)<<4)+b2;
     if(diff&0x20) diff-=0x40+4; else diff+=4;
     if(bcount==1) b1=c1&0x0F;//zwar 2 gebraucht aber auch 2 nachgeladen
     else bcount-=2;//2Nibbles verbraucht
     wert+=diff;
    }
   else //if((b1&0xC)==0)
    {if(format==1)
      {c1=getc(fp);
       if(bcount==1) {b2=(c1>>4)&0x0F; b3=c1&0x0F;}
       else b3=(c1>>4)&0x0F;
       wert=(b1<<8)+(b2<<4)+b3;
       if(--bcount==1) b1=c1&0x0F;//noch 1Nibble uebrig
      }
     else //format==2
      {c1=getc(fp);
       if(bcount==1)
        {c2=getc(fp); b2=(c1>>4)&0x0F; b3=c1&0x0F; b4=(c2>>4)&0x0F;}
       else //bcount==2
        {b3=(c1>>4)&0x0F; b4=c1&0x0F;}
       wert=(b1<<12)+(b2<<8)+(b3<<4)+b4;
       if(bcount==1) b1=c2&0x0F;//noch 1Nibble uebrig
       else bcount=0;//kein Nibble mehr uebrig
      }
    }
  }
  if(format==2) y=wert*0.1;
  else y=umrechnen(sensor,variante,wert);
  if(argflag['S']) x2=x/3600.0;
  else if(argflag['M']) x2=x/60.0;
  else x2=x;
  if(argflag['V']) printf("x=%g value=%d y=%lf\n",x2,wert,y);
  fprintf(fp2,"%g %g\n",x2,y);
 }
}

double umrechnen(int sensor,int variante,int wert)
{
 const double r1=2000;//2kOhm Widerstand nach AVCC, Messwiderstand nach GND
 const double wmax=1024;//Maximalwert 10-Bit-AD-Wandler + 1
 double r2,y;
 int i;
 static int f=1;
 if(sensor==222 && variante==1) //Temperaturfuehler
  {
   r2=r1*wert/(wmax-wert);
   if(argflag['V']) printf("r2=%lf Ohm\n",r2);//test
   /* provisorische Variante:
      1619 Ohm --> 0 Grad
      3426 Ohm --> 100 Grad
      Rest linear intrapoliert (Abweichungen etwa 2-3 Grad)
   */
   //y=100.0/(3426-1619)*(r2-1619);

   /* genauere Variante:
      Position in Tabelle suchen und
      zwischen entsprechenden 2 Werten intrapolieren
   */
   static int tabelle[]=
    {-55, 990, -50, 1040, -40, 1146, -30, 1260, -20, 1381, -10, 1510, 0, 1646, 10, 1790,
     20, 1941, 25, 2020, 30, 2100, 40, 2267, 50, 2441, 60, 2623, 70, 2812, 80, 3009,
     90, 3214, 100, 3426, 110, 3643, 120, 3855, 125, 3955, 130, 4048, 140, 4208, 150, 4323,
     0, 0
     };
   int t0,t1;
   double w0,w1;
   for(i=0;tabelle[i+5]!=0 && r2>tabelle[i+3];) i+=2;
   t0=tabelle[i]; t1=tabelle[i+2];
   w0=tabelle[i+1]; w1=tabelle[i+3];
   y=t0+(t1-t0)*(r2-w0)/(w1-w0);
  }
 else if(sensor==83 && variante==1) //Phototransistor
  {
   if(argflag['V'] && f>0)
    {printf("Sensor = Phototransistor, Value of 1023 represents about 2.56 Volts\n"); --f;}
   y=wert;
  }
 else //unbekannter Sensor oder Variante
  {
   if(argflag['V'] && f>0)
    {printf("unbekannter Sensor oder unbekannte Anschlussvariante\n"); --f;}
   y=wert;
  }
 return y;
}

/**
int rueckumrechnen(int sensor,int variante,double y)
{
 const double r1=2000;//2kOhm Widerstand nach AVCC, Messwiderstand nach GND
 const double wmax=1024;//Maximalwert 10-Bit-AD-Wandler + 1
 double r2;
 int wert;
 if(sensor==222 && variante==1)
  {
   // provisorische Variante:
   // 1619 Ohm --> 0 Grad
   // 3426 Ohm --> 100 Grad
   // Rest linear intrapoliert

   r2=y*(3426-1619)/100.0+1619;
   if(argflag['V']) printf("r2=%lf\n",r2);//test
   wert=idfix(r2*wmax/(r1+r2));
  }
 else if(sensor==83 && variante==1) //Phototransistor
  {
   wert=idfix(y);
  }
 else //unbekannter Sensor oder Variante
  {static int f=1;
   if(argflag['V'] && f>0)
    {printf("unbekannter Sensor oder unbekannte Anschlussvariante\n"); --f;}
   wert=idfix(y);
  }
 return wert;
}
**/
