/* lcdanzeige.cc  Ansteuern einer einfachen LCD-Anzeige
Schaltschema:
  Port D4,D5,D6,D7 verbunden mit 32-Bit Schieberegister bestehend aus 4 CD4094:
  D4=CLK, D5=OE, D6=D (Daten-Eingang), D7=STR
  Die Ausgaenge des Schieberegisters sind mit den Segmenten eines
  3 1/2 stelligen LCDs verbunden:
    A0=3DP, A1=3E, A2=3D, A3=3C, A4=2DP, A5=2E, A6=2D, A7=2C,
    A8=1DP, A9=1E, A10=1D, A11=1C, A12=1B, A13=1A, A14=1F, A15=1G,
    A16=2B, A17=2A, A18=2F, A19=2G, A20=Pin28, A21=3B, A22=3A, A23=3F,
    A24=3G, A25=Pin38, A26=Pin39, A27=Pin3, A28=Pin2, A29=COM, A30=NC, A31=NC.
  Dabei ist COM der gemeinsame GND der Anzeige. Dieser darf nicht direkt mit
  GND verbunden sein weil die Anzeige einen Polaritaetswechsel mit etwa 20Hz
  haben sollte.  A-F sind die 7 Segmente der Ziffern:
    A
  +----+    1 ist Ziffer ganz rechts, DP ist jeweils Punkt links der Ziffer.
  |    |    2 ist zweite Ziffer von rechts
 F| G  |B   Pin28 ist in aktuller LCD nicht belegt. Andere Typen haben hier
  +----+    einen Doppelpunkt links der Ziffer2.
  |    |    Pin38 = Over
 E|    |C   Pin39 = - (Minuszeichen)
  +----+    Pin3  = Segmente A+B der 4.Ziffer (Darstellung der fuehrenden 1)
    D       Pin2  = Batt

Funktionsweise des Schieberegisters:
  Das aktuelle Bit wird auf D bereitgestellt, dann CLK kurz (etwa 1usec) auf 1
  gesetzt und dann wieder 0. So werden alle 32 Bits hineingeschoben. Das erste
  hineingeschobene Bit wird zu A31, das letzte zu A0.
  Mit Setzen von STR auf 1 werden die Daten auf die Ausgaenge uebernommen.
  OE kann immer auf 1 bleiben. Mit 0 auf OE bleibt die Anzeige leer.

Autor: Rolf Pfister
Copyright: Freeware

History:
13.5.2009	Erstellung

*/

#include <avr/io.h>
#include "ulong.h"


/*
static uchar jwait;
void microwait(uchar n)
{
 uchar j;
 for(jwait=j=0;j<n;j++) jwait+=j;
}
*/

void datenschieben(ulong n)
{
 uchar c;
 for(int i=0;i<32;i++,n<<=1)
  {PORTD = c = 0x20;//OE, vorhergehende Daten immer noch anzeigen
   if(n&0x80000000) c |= 0x40; //zu schiebendeS Datenbit
   PORTD = c;
   c |= 0x10;//CLK
   PORTD = c;
  }
 PORTD = c |= 0x80;//STR
 // microwait(1);//etwa 1 usec warten
 PORTD=0x20;//OE
}

const ulong SEG3DP=1,SEG3E=2,SEG3D=4,SEG3C=8,SEG2DP=0x10,SEG2E=0x20,SEG2D=0x40,
  SEG2C=0x80,SEG1DP=0x100,SEG1E=0x200,SEG1D=0x400,SEG1C=0x800,SEG1B=0x1000,
  SEG1A=0x2000,SEG1F=0x4000,SEG1G=0x8000,SEG2B=0x10000,SEG2A=0x20000,
  SEG2F=0x40000,SEG2G=0x80000,SEGPin28=0x100000,SEG3B=0x200000,SEG3A=0x400000,
  SEG3F=0x800000,SEG3G=0x1000000,SEGPin38=0x2000000,SEGPin39=0x4000000,
  SEGPin3=0x8000000,SEGPin2=0x10000000,SEGCOM=0x20000000;
const ulong
  SEG4BC=SEGPin3,SEGMinus=SEGPin39,//kann bei andern LCDs anders sein
  SEGBatt=SEGPin2,SEGOver=SEGPin38;//kann bei andern LCDs anders sein
  //SEGDoppelpunkt=SEGPin28;//bei diesem LCD nicht vorhanden
static ulong siebensegdec1[10]={
 SEG1A+SEG1B+SEG1C+SEG1D+SEG1E+SEG1F, //0
 SEG1B+SEG1C, //1
 SEG1A+SEG1B+SEG1D+SEG1E+SEG1G, //2
 SEG1A+SEG1B+SEG1C+SEG1D+SEG1G, //3
 SEG1B+SEG1C+SEG1F+SEG1G, //4
 SEG1A+SEG1C+SEG1D+SEG1F+SEG1G, //5
 SEG1A+SEG1C+SEG1D+SEG1E+SEG1F+SEG1G, //6
 SEG1A+SEG1B+SEG1C, //7
 SEG1A+SEG1B+SEG1C+SEG1D+SEG1E+SEG1F+SEG1G, //8
 SEG1A+SEG1B+SEG1C+SEG1D+SEG1F+SEG1G}; //9
static ulong siebensegdec2[10]={
 SEG2A+SEG2B+SEG2C+SEG2D+SEG2E+SEG2F, //0
 SEG2B+SEG2C, //1
 SEG2A+SEG2B+SEG2D+SEG2E+SEG2G, //2
 SEG2A+SEG2B+SEG2C+SEG2D+SEG2G, //3
 SEG2B+SEG2C+SEG2F+SEG2G, //4
 SEG2A+SEG2C+SEG2D+SEG2F+SEG2G, //5
 SEG2A+SEG2C+SEG2D+SEG2E+SEG2F+SEG2G, //6
 SEG2A+SEG2B+SEG2C, //7
 SEG2A+SEG2B+SEG2C+SEG2D+SEG2E+SEG2F+SEG2G, //8
 SEG2A+SEG2B+SEG2C+SEG2D+SEG2F+SEG2G}; //9
static ulong siebensegdec3[10]={
 SEG3A+SEG3B+SEG3C+SEG3D+SEG3E+SEG3F, //0
 SEG3B+SEG3C, //1
 SEG3A+SEG3B+SEG3D+SEG3E+SEG3G, //2
 SEG3A+SEG3B+SEG3C+SEG3D+SEG3G, //3
 SEG3B+SEG3C+SEG3F+SEG3G, //4
 SEG3A+SEG3C+SEG3D+SEG3F+SEG3G, //5
 SEG3A+SEG3C+SEG3D+SEG3E+SEG3F+SEG3G, //6
 SEG3A+SEG3B+SEG3C, //7
 SEG3A+SEG3B+SEG3C+SEG3D+SEG3E+SEG3F+SEG3G, //8
 SEG3A+SEG3B+SEG3C+SEG3D+SEG3F+SEG3G}; //9

void zahlauflcdanzeigen(int zahl,int teiler,int polaritaet)
{
 //Beispiel: zahl=1234 teiler=100 --> Anzeige="12.34"
 ulong n;
 int z1,z2,z3;
 switch(teiler)
  {case 10: n = SEG1DP; break;
   case 100: n = SEG2DP; break;
   case 1000: n = SEG3DP; break;
   default: n=0;
  }
 if(zahl<0) {zahl= -zahl; n |= SEGMinus;}
 if(zahl>1999) zahl=1999; //Fehler: darstellbare Zahlen auf 1999 beschraenkt
 z3=zahl/100;
 z2=(zahl-100*z3)/10; //in Assembler schneller als z2=(zahl/10)%10;
 z1=zahl-100*z3-10*z2; //in Assembler schneller als z1=zahl%10;
 if(zahl>999)
  {z3 -= 10; //einfacher und schneller als z3 %= 10;
   n |= SEG4BC;
  }
 n += siebensegdec3[z3] + siebensegdec2[z2] + siebensegdec1[z1];
 if(polaritaet!=0) n ^= 0xFFFFFFFF;
 datenschieben(n);
}

int main(void)
{
 int i,j,zahl=990;
 DDRD = 0xF0; //PortD obere 4 Bits als Ausgaenge
 PORTD= 0x0C; //D2,D3 Pullupwiderstaende setzen (fuer Taster-Eingaenge)

 while(1) //Hauptschlaufe
  {for(j=0;j<100;j++)
    {for(i=0;i<20;i++) zahlauflcdanzeigen(zahl,100,0);
     for(i=0;i<20;i++) zahlauflcdanzeigen(zahl,100,1);
    }
   zahl = -zahl; //Vorzeichen in etwa im Sekundentakt wechseln
   if(zahl>=0) zahl++;
  }
 return 0;
}
