;-----------------------------------------------------------------------------
;* Thermologger.asm            letzte Aenderung: 20.4.2009
;* Version: 0.1
;* Prozessor: Atmega8, 8MHz
;* Schaltschema: thermologger.png, Port B0,B1,B2 mit LEDs verbunden
;*               an PD2 (Pin 4) Taster, wenn gedrueckt == 0
;*   Variante 1: ADC0 (Pin 23) Thermowiderstand gegen GND, 2kOhm gegen AVCC
;*   Variante 2: ADC0 (Pin 23) Thermowiderstand gegen GND, 3.9kOhm gegen AVCC
;* Autor: Rolf Pfister
;* Copyright: Freeware
;* History:
;* 11.4.2009	Erstellung
;* 17.4.        Mit Packen der Daten (Version 0.1)
;* 
;-----------------------------------------------------------------------------
.include "m8def.inc" 	; Definitionsdatei fuer den Prozessortyp

.equ	SENSORTYP=222		;Temperatursensor KTY81-222
.equ	ANSCHLUSSVARIANTE=1	;Anschluss ueber 2kOhm, und AREF=AVCC
;.equ	ANSCHLUSSVARIANTE=2	;Anschluss ueber 3.9kOhm, und AREF=Intern 2.56V
;.equ	DATENFORMAT=0		;0=ungepackt = 2 Byte pro Messpunkt
.equ	DATENFORMAT=1		;1=Einfaches packen = 0.5 bis 2 Byte pro Mess.

;-----------------------------------------------------------------------------
;Datenformat:
; 1 Byte DATENFORMAT
; 1 Byte SENSORTYP
; 1 Byte ANSCHLUSSVARIANTE
; 1 Byte Messperiode = Anzahl Sekunden zwischen 2 Messungen
; 2 Byte N = Anzahl Messpunkte
;Wenn DATENFORMAT==0: N*2 Byte ungepackte Daten
;sonst je nach Packvariante weniger als N*2 Bytes
;
;DATENFORMAT==1:
; Wenn erstes Bit gesetzt, dann sind die naechsten 3 Bits
; die Differenz zum vorherigen Wert (Bereich -4 bis +3).
; Wenn das erste Bit nicht, aber das 2.Bit gesetzt ist,
; dann sind die naechsten 6 Bits +-4 die Differenz zum vorherigen Wert.
; (+-4 heisst: wenn Diff. positiv noch 4 addieren, wenn negativ 4 subtrahieren)
; Somit Wertebereich 4 bis 35 und -5 bis -36.
; Wenn weder das erste noch das 2. Bit gesetzt sind (c & 0xC0 == 0), dann
; sind die naechsten 10 Bits ein absoluter Wert.
; Also kurz zusammengefasst:
;       1X -> 4Bit, davon 3Bit Differenzwert (-4 ... +3)
;       01 -> 8Bit, davon 6Bit Differenzwert (-36 ... -5 und +4 ... +35)
;       00 -> 12Bit, davon 10Bit Absolutwert vorzeichenlos (0 ... 1023)
;
;-----------------------------------------------------------------------------
;Speicherbelegung:
;Global benutzt:
; r11:r10 Millisekunden, r12 Sekunden, r13 Minuten, r14 Stunden, r15 Tage
; r4=Flag fuer AD-Wandlung
; r5: Vorheriges Nibble beim gepackten Speichern (Gueltig=r5&0x10)
; r7:r6 letzter Messpunkt
; r23:r22 Restlicher Platz in Byte
; r25:r24 Anzahl bisher gespeicherte Messpunkte
; r27:r26 == XH:XL = Zeiger auf zu speichernde Daten
; r29:r28 == YH:YL = Zeiger auf am Schluss zu speichernde Anzahl Messpunkte
; 
;im Hauptprogramm:
; r17:r16 fuer kurze Wartezeit (immer innerhalb 1...255)
; r16 auch als temporaere Variable
; r1=Messperiode Anzeige mit LEDs, 1=rot 2=gruen 4=gelb
; r2=Messperiode in Sekunden
; r3=1
; r19:r18 Millisekunden (eventuell falls noch Uhr benoetigt)
; r20,r21 Sollzeit
; 
;-----------------------------------------------------------------------------

; Reset und Interruptvectoren	;VNr. Beschreibung
begin:	rjmp	main		; 1   Power On Reset
	reti			; 2   Int0-Interrupt
	reti			; 3   Int1-Interrupt
	reti			; 4   TC2 Compare Match
	reti			; 5   TC2 Overflow
	reti			; 6   TC1 Capture
	reti			; 7   TC1 Compare Match A
	reti			; 8   TC1 Compare Match B
	reti			; 9   TC1 Overflow
	rjmp tc0over		; 10  TC0 Overflow
	reti			; 11  SPI, STC = Serial Transfer Complete
	reti			; 12  UART Rx Complete
	reti			; 13  UART Data Register Empty
	reti			; 14  UART Tx Complete
	rjmp adcomplete		; 15  ADC Conversion Complete
	reti			; 16  EEPROM ready
	reti			; 17  Analog Comperator
	reti			; 18  TWI (I2C) Serial Interface
	reti			; 19  Store Program Memory ready
;-----------------------------------------------------------------------------
; Start, Power On, Reset
main:	ldi r16, RAMEND&0xFF
        out SPL, r16  	     ; Init Stackpointer L
	ldi r16, RAMEND>>8
        out SPH, r16  	     ; Init Stackpointer H

;; Init-Code
	ldi r16, 0xFF		;Alle Leitungen auf Ausgang
        out DDRB, r16		;Data Direction Register B setzen
	ldi r16, 0
	out PORTB, r16		;alle LEDs loeschen
        ldi r17, 0x00
        out DDRD, r17		;Alle Leitungen Port D auf Eingang
        ldi r17, 0b00001100	;D2,D3 Pullupwiederstaende setzen
        out PORTD, r17	        ;PortD Pullupwiederstaende setzen
	
	ldi r17, 0x01
	out TIMSK, r17		;Timer/Counter0 Overflow einschalten
	ldi r17, 3		;1=nicht teilen, 2=durch 8, 3=64, 4=256, 5=1024
	out TCCR0, r17		;Vorteiler auf 64 setzen
;Bei 8MHz wird somit in 8us Schritten gezaehlt (bei 3.6864MHz etwa 8*2.17us)
;Um alle 1000usec einen Interrupt zu erhalten muss der Zaehler jeweils bei
;256-125=131 gestartet werden.
;(bei 3.6864MHz 256-57.6=198.4 oder jeweils 2*57 und 3*58 us)
.equ	ZAEHLER =131		; 131 bei 8MHz, 198 oder 199 bei 3.6864MHz
	ldi r17, ZAEHLER
	out TCNT0, r17
	clr r10
	clr r11			; r11r10 = Millisec
	clr r12			; Sekunden
	clr r13			; Minuten
	clr r14			; Stunden
	clr r15			; Tage, auf 0 setzen
	sei			;Interrupt erlauben

; AD-Wandler initialisieren:
	clr r4			;Speichern-Flag fuer AD-Interrupt-Routine
;zum Austesten mit SRAM:
;	ldi XH, 0
;	ldi XL, 0x60		;provi.: X auf Anfang des SRAM setzen
	ldi XH, 0
	ldi XL, 0		;X auf Anfang im EEPROM setzen
	ldi r16, 0x40		;ADC0 und Referenz=AVCC (mit externem Condenser)
;Anschlussvariante2:
;	ldi r16, 0xC0		;ADC0 und Referenz=2.56V (ev.externer Condenser)
	out ADMUX,r16
;Prescaler: 5=32 6=64 7=128, 8MHz/64=125kHz (muss im Bereich 50-200 liegen)
.equ	PRESCALER=6
;r16 = Prescaler+0b11001000 (ADEN+ADSC+ADIE = Enable,Einzelstart,Interrupt)
;ADIF (0x10) wird automatisch vom AD-Wandler gesetzt wenn Wandlung fertig.
	ldi r16, 0b11001000 + PRESCALER
	out ADCSRA, r16		;Prescaler und Enable

	clr r1			;Start-Parameter
	inc r1			;Start mit Roter LED
	mov r3, r1		;r3=1
;vorlaeufig fuer EEPROM: Atmega8 hat 512 Bytes = 256 WORDs EEPROM
.if DATENFORMAT==0
	ldi r22, 250		;Kopfdaten = 512-6 Bytes
	ldi r23, 1		;r23:r22 = 256+250 = 506
.else ;if DATENFORMAT==1
	ldi r22, 248	;bei gepacktem Speichern noch 2 Reserve
			;zum am Schluss noch ev. letztes Nibble zu speichern
			;(2 weil manchmal gleich 2 Byte gespeichert werden)
	ldi r23, 1	;r23:r22 = 256+248 = 504
.endif
	clr r17			;r17=0
	clr r5			;Nibble fuer gepacktes Speichern loeschen
;-----------------------------------------------------------------------------
startloop:
	out PORTB, r1		;ausgewaehlte LED leuchten lassen
	sbic PIND, 2		;Skip if Bit Clear, Taster gedrueckt?
	rjmp startloop		;nein--> warten
	ldi r16, 200		;r17:r16=200
	rcall milliwait		;200ms warten
	sbis PIND, 2		;Skip if Bit Set, Taster losgelassen?
	rjmp evlangertastendruck	;nein--> langer Tastendruck
kurzertastendruck:
	rol r1			;naechste LED
	sbrc r1,3		;0x08 erreicht ?
	mov r1,r3		;ja: r1=1
	rjmp startloop
evlangertastendruck:	
	ldi r16, 250
	rcall milliwait		;250ms warten
	sbic PIND, 2		;Skip if Bit Clear, Taster noch gedrueckt?
	rjmp kurzertastendruck	;nein-> doch nur kurzer Tastendruck
langertastendruck:
	out PORTB, r17
	ldi r16, 250
	rcall milliwait		;250ms warten
	out PORTB, r1
	rcall milliwait		;250ms warten
	sbis PIND, 2		;Skip if Bit Set, Taster losgelassen?
	rjmp langertastendruck	;nein--> warten
aufganzesekundewarten:
;	cli			;Interrupt verbieten
;	mov r18, r10
;	mov r19, r11		;r19:r18 = Startzeit Millisekunden
;	sei			;Interrupt erlauben
;	ldi r16, 0xE8
;	ldi r17, 0x03		;r17:r16 = 1000
;	sub r16, r18
;	sbc r17, r19		;r17:r16 -= aktuelle Millisekunden
;	rcall milliwait		;auf ganze Sekunde warten
;	clr r17
	clr r10			;Millisekunden auf 0 setzen
	clr r11			;ok solange wir keine Echtzeit-Uhr brauchen
	mov r20, r12		;r20 = Startzeit Sekunden
	mov r21, r13		;r21 = Startzeit Minuten

; r1 --> r2 = Messperiode in Anzahl Sekunden:
	sbrc r1, 0		;Skip if Bit in Register Clear, Rot gewaehlt?
	rjmp auswahl1		;ja-->
	sbrc r1, 1		;Skip if Bit in Register Clear, Gruen gewaehlt?
	rjmp auswahl2		;ja-->
auswahl3:
	ldi r16, 60		;60 Sekunden Messperiode
	rjmp auswahlr2
auswahl1:
	ldi r16, 1		;1 Sekunde Messperiode
	rjmp auswahlr2
auswahl2:
	ldi r16, 10		;10 Sekunden Messperiode
auswahlr2:
	mov r2, r16
	
;Datenkopf schreiben:
	ldi r16, DATENFORMAT
	rcall eepromstore
	ldi r16, SENSORTYP
	rcall eepromstore
	ldi r16, ANSCHLUSSVARIANTE
	rcall eepromstore
	mov r16, r2
	rcall eepromstore	;Messperiode in Sekunden
	clr r24
	clr r25			;Anzahl Messpunkte = 0
	mov YH,XH		;Y = Zeiger zum am Schluss Anzahl
	mov YL,XL		;Messpunkte zu speichern.
	ldi r16, 0xFF
	rcall eepromstore	;Anzahl Messpunkte erst mal auf ungueltigen
	rcall eepromstore	;Wert (0xFFFF) setzen.
;	clr r6			;Letzter Messwert auf 0 setzen nicht noetig
;	clr r7			;weil erster Messwert eh Absolutwert.
messloop:
	rcall messung		;AD-Wandlung machen
	add r20, r2		;neue Sollzeit = alte Sollzeit + Messperiode
	cpi r20, 60		;r20<60 ?
	brlt warteloop		;ja-->
	subi r20, 60		;nein: 60 subtrahieren
	;braucht da noch Anpassung falls Messperiode groesser 60 sein kann!
	inc r21			;und Minuten erhoehen
	cpi r21, 60
	brne warteloop
	clr r21
warteloop:
	sbis PIND, 2		;Skip if Bit Clear, Taster gedrueckt?
	rjmp evvorzeitig	;ja--> ev. vorzeitig beenden
	cp r20, r12		;Sollzeit Sekunden erreicht?
	brne warteloop
	cp r21, r13		;Sollzeit Minuten erreicht?
	brne warteloop
	rcall istspeichervoll
	brne messloop
	rjmp mfertig
evvorzeitig:
	ldi r16, 255
	rcall milliwait		;255ms warten
	sbis PIND, 2		;Skip if Bit Set, Taster immer noch gedrueckt?
	rcall milliwait		;ja: nochmals 255ms warten
	sbic PIND, 2		;Taster immer noch gedrueckt?
	rjmp warteloop		;nein--> dochnicht
mfertig:tst r4
	brne mfertig		;allenfalls warten bis laufende Messung fertig
	sbrs r5, 4		;Bit 4 in r5 gesetzt?
	rjmp mfert2		;nein --> kein vorheriges Nibble
	mov r16, r5
	swap r16
	andi r16, 0xF0		;nicht unbedingt noetig (unteres Nibble wird
				;eh verworfen), aber leichter zum debuggen.
	rcall eepromstore	;letztes Nibble auch noch speichern
mfert2:	mov XH, YH
	mov XL, YL
	mov r16, r25
	rcall eepromstore
	mov r16, r24
	rcall eepromstore	;Anzahl Messpunkte speichern
halteloop:
	rol r1
	sbrc r1, 3		;0x08 erreicht ?
	mov r1, r3		;ja: r1=1
	out PORTB, r1
	ldi r16, 255
	rcall milliwait		;255ms warten
	rjmp halteloop


eepromstore2:		;Byte speichern und den Zaehler r23:r22 erniedrigen
	subi r22, 1
	sbci r23, 0
	cpi r23,0xFF
	brne eepromstore
	cpi r22,0xFF	;Speicher voll? (sollte eigentlich nicht passieren)
	breq eeret	;ja --> nichts machen
eepromstore:
;	st X+, r16	;fuer test mit SRAM
;	ret		;fuer test mit SRAM
eeL1:	sbic EECR,1
	rjmp eeL1	;warte bis EEPROM bereit
	out EEARH,XH
	out EEARL,XL
	out EEDR, r16
	sbi EECR, 2	;EEMWE-Bit setzen
	sbi EECR, 1	;EEWE-Bit setzen
	adiw XL, 1
eeret:	ret

;-----------------------------------------------------------------------------
; Interrupt-Routine
; Registerbelegungen:
; r11:r10 Millisekunden, r12 Sekunden, r13 Minuten, r14 Stunden, r15 Tage
tc0over:push r16
	push r17
	in r17, SREG	;Statusregister sichern
	push r17
	ldi r17, ZAEHLER
	out TCNT0, r17
	ldi r16, 0xE8
	ldi r17, 0x03	;r17:r16 = 1000 = 0x03E8
	inc r10		;Millisekunden erhoehen
	brne L1
	inc r11
L1:	cp r10, r16
	brne end1	;fertig wenn msec!=1000
	cp r11, r17
	brne end1	;fertig wenn msec!=1000
	clr r10
	clr r11		;Millisekunden auf 0 setzen
	inc r12		;Sekunden erhoehen
	ldi r16, 60
	cp r12, r16
	brne end1	;fertig wenn sec!=60
	clr r12		;Sekunden auf 0 setzen
	inc r13		;Minuten erhoehen
	cp r13, r16
	brne end1	;fertig wenn min!=60
	clr r13		;Minuten auf 0 setzen
	inc r14		;Stunden erhoehen
	ldi r16, 24
	cp r14, r16
	brne end1	;fertig wenn std!=24
	clr r14		;Stunden auf 0 setzen
	inc r15		;Tage erhoehen
end1:	pop r17
	out SREG, r17	;Statusregister zurueckholen
	pop r17
	pop r16
	reti

;-----------------------------------------------------------------------------
;Interrupt-Routine fuer AD-Wandlung
; Registerbelegungen:
; r27:r26 == XH:XL = Zeiger auf zu speichernde Daten
; r4 = Datenspeichern-Flag, Rueckgesetzt wenn AD-Wandlung fertig
; r25:r24 Anzahl gespeicherte Messpunkte += 1
adcomplete:
	push r16
	push r17
	in r17, SREG	;Statusregister sichern
	push r17
	tst r4		;allenfalls erste Wandlung verwerfen
	breq end2
	in r17, ADCL	;L muss zuerst gelesen werden!
	in r16, ADCH
	sei		;Interrupts erlauben, noetig weil EEPROM eventuell
			;laenger braucht (und Uhr weiter laufen muss)
	;(ok EEPROM braucht nicht soo lange, aber beim FLASH wirklich noetig)
.if DATENFORMAT==0
	rcall eepromstore2 	;H zuerst speichern
	mov r16, r17
	rcall eepromstore2	;L danach speichern
.else ;if DATENFORMAT==1
	rcall gepacktspeichern
.endif
	adiw r24, 1		;r25:r24 += 1
	clr r4
	out PORTB, r22		;test
end2:	pop r17
	out SREG, r17	;Statusregister zurueckholen
	pop r17
	pop r16
	reti

;-----------------------------------------------------------------------------
; Subroutine gepacktspeichern(UWORD r16:r17)
; r25:r24 = Anzahl bisherige Messpunkte, unveraendert
; r5 = vorheriges Nibble falls Bit 4 gesetzt, wird aktualisiert
; r7:r6 = alter Messwert, wird dann aktualisiert
; r16:r17 = neuer Messwert, geht verloren, ja wirklich r16=H r17=L
gepacktspeichern:
	push r1
	push r2
	mov r1, r16
	mov r2, r17		;Aktueller Messwert: r1:r2 = r16:r17
	tst r24			;Anzahl bisherige Messpunkte == 0 ?
	brne packstart		;nein -->
	tst r25
	breq pack0		;ja --> ungepackter Absolutwert
packstart:
;       1X -> 4Bit, davon 3Bit Differenzwert (-4 ... +3)
;       01 -> 8Bit, davon 6Bit Differenzwert (-36 ... -5 und +4 ... +35)
;       00 -> 12Bit, davon 10Bit Absolutwert (0 ... 1023)
	sub r17, r6	;r7:r6 = alter Wert, (r1:r2 = neuer Wert)
	sbc r16, r7	;r16:r17 = Differenz
	tst r16
	breq pack9	;positive Differenz weniger als 9 Bit -->
	cpi r16, 0xFF	;negative Differenz weniger als 9 Bit ?
	brne pack0	;nein --> ungepackter Absolutwert
pack9:	sbrs r17, 7	;Differenz negativ?
	rjmp pack9p	;nein -->
	cpi r17, 0xFC	;passt negativer Wert in 3 Bit?
	brcc pack1	;ja --> in 4 Bit = 1Nibble packen
	subi r17, -4	;Diff.Wertebereich erhoehen
	cpi r17, 0xE0	;passt negativer Wert in 6 Bit?
	brcs pack0	;nein --> Absolutwert speichern
	rjmp pack2	;ja --> in 2 Nibbles packen
pack9p:	cpi r17, 0x04	;passt positiver Wert in 3 Bit?
	brcs pack1	;ja --> in 1Nibble packen
	subi r17, 4	;Diff.Wertebereich erhoehen
	cpi r17, 0x20	;passt positiver Wert in 6 Bit?
	brcc pack0	;nein --> Absolutwert speichern
pack2:			;in 8 Bit = 2Nibble packen
	andi r17,0x3F	;6 Nutzbits
	ori r17, 0x40	;01 als erste 2 Bits
	sbrs r5, 4	;Bit 4 in r5 gesetzt?
	rjmp pack2k	;nein --> kein vorheriges Nibble
	swap r5		;vorheriges Nibble zuerst..
	mov r16, r5
	andi r16, 0xF0	 ;und zusammen mit..
	push r17
	swap r17
	andi r17, 0x0F 	  ;erstem Nibble..
	add r16, r17
	pop r17
	rcall eepromstore2  ;abspeichern.
	andi r17,0x0F		;zweites Nibble
	ori r17, 0x10		;als gueltig markieren
	mov r5, r17		;und nach r5 kopieren.
	rjmp packret
pack2k:	mov r16, r17		;genau 1 Byte speichern
	rcall eepromstore2
	rjmp packret
pack1:	;in 4 Bit packen
	andi r17,0x07
	ori r17, 0x08
	sbrs r5, 4		;Bit 4 in r5 gesetzt?
	rjmp pack1k		;nein --> kein vorheriges Nibble
	swap r5			;ja: vorheriges Nibble einfuegen
	mov r16, r5
	andi r16, 0xF0
	add r16, r17
	rcall eepromstore2
	clr r5			;r5 als ungueltig markieren
	rjmp packret
pack1k:	ori r17, 0x10		;als gueltig markieren
	mov r5, r17		;und in r5 kopieren
	rjmp packret
pack0:				;12Bit Absolutwert (=r1:r2) speichern
	sbrs r5, 4		;Bit 4 in r5 gesetzt?
	rjmp pack03		;nein --> kein vorheriges Nibble
	;andi r1, 0x03	;nur noetig wenn obere Bits in r1 nicht schon 0 waeren.
	mov r16, r5		;ja: Nibble r5 -> oberes Nibble r16:r17
	swap r16		;r16 = r5<<4
	andi r16, 0xF0
	add r16, r1
	rcall eepromstore2 	;2 obere Nibbles zuerst speichern
	mov r16, r2
	rcall eepromstore2	;2 untere Nibbles danach speichern
	rjmp packret
pack03:				;3 Nibbles speichern
	mov r7, r1
	mov r6, r2		;letzter Messwert r7:r6 = r1:r2 = neuer Wert
	ldi r16, 4
paL1:	lsl r2
	rol r1
	dec r16
	brne paL1		;r1:r2 um 4 Bit nach links schieben
	mov r16, r1
	rcall eepromstore2	;erste 2 Nibbles wirklich speichern
	mov r5, r2
	swap r5			;drittes Nibble im r5
	set			;T-Flag setzen
	bld r5, 4		;in r5 Bit4=T == r5 als gueltig markieren
	rjmp packret2
packret:mov r7, r1
	mov r6, r2		;letzter Messwert r7:r6 = r1:r2 = neuer Wert
packret2:
	pop r2
	pop r1
	ret

;-----------------------------------------------------------------------------
milliwait:			;Wartezeit: r17:r16 msec, min 1 max 999
	push r16
	push r17
	push r18
	push r19
	mov r18, r10
	mov r19, r11		;r19r18 = sollzeit = aktuelle Millisec
	add r18, r16
	adc r19, r17		;sollzeit += Wartezeit
	ldi r16, 0xE8		;auf 1000 testen
	cpi r19, 0x03		;oberes Byte testen
	brcs loop1		;sollzeit<1000 --> weiter
	brne subtra		;sollzeit>1000 --> subtrahiere 1000
	cp r18, r16		;unteres Byte testen
	brcs loop1		;sollzeit<1000 --> weiter
subtra:	sub r18, r16
	sbci r19, 0x03		;wenn sollzeit>=1000 dann sollzeit-=1000
loop1:	cp r18, r10
	brne loop1
	cp r19, r11		;auf sollzeit warten
	brne loop1
	pop r19
	pop r18
	pop r17
	pop r16
	ret
;-----------------------------------------------------------------------------
; Messung machen
messung:
	push r16
	push r17
	push r18
	ldi r16, 100
	clr r17			;r17:r16 = 100
mess1:	tst r4
	brne mess1		;ev. warten bis vorherige AD-Wandlung fertig
	inc r4			;Flag fuer naechste Wandlung setzen
	sbi ADCSRA,6		;Starte naechste AD-Wandlung
	ldi r18, 3		;LED 3 mal blinken
	rjmp blink
blinkloop1:
	rcall milliwait		;100ms warten
blink:	out PORTB, r1
	rcall milliwait		;100ms warten
	out PORTB, r17
	dec r18
	brne blinkloop1
	pop r18
	pop r17
	pop r16
	ret

istspeichervoll:
	tst r23
	brne return
	tst r22			;restlicher Platz == 0?
return:	ret			;Z-Flag gesetzt wenn Speicher voll
